from __future__ import annotations

from dataclasses import dataclass
from typing import List, Optional, Tuple

from bot.models import BoardMember
from bot.storage import BoardRepository, get_default_repo


ROLE_OWNER = "owner"
ROLE_PRODUCT_OWNER = "po"
ROLE_SCRUM_MASTER = "sm"
ROLE_DEVELOPER = "dev"
ROLE_VIEWER = "viewer"

ALL_ROLES = [ROLE_OWNER, ROLE_PRODUCT_OWNER, ROLE_SCRUM_MASTER, ROLE_DEVELOPER, ROLE_VIEWER]


@dataclass
class MemberWithRole:
    user_id: int
    display_name: str
    role: str
    is_owner: bool


class RolesService:
    """مدیریت اعضا و نقش‌ها، با فرض یک owner به‌ازای هر گروه."""

    def __init__(self, repo: BoardRepository | None = None) -> None:
        self.repo: BoardRepository = repo or get_default_repo()

    def _display_name_from_parts(self, first_name: str | None, last_name: str | None,
                                 username: str | None) -> str:
        parts = []
        if first_name:
            parts.append(first_name)
        if last_name:
            parts.append(last_name)
        name = " ".join(parts).strip()
        if not name and username:
            name = f"@{username}"
        if not name:
            name = "کاربر"
        return name

    def ensure_member_from_update(self, chat_id: int, user: dict) -> MemberWithRole:
        """هر بار پیامی می‌آید، مطمئن می‌شویم عضو در برد ثبت شده است.

        اگر هنوز هیچ owner ای برای این chat ثبت نشده باشد، همان کاربر را owner می‌کنیم.
        """
        user_id = user.get("id")
        if user_id is None:
            raise ValueError("user id is required")
        first_name = user.get("first_name")
        last_name = user.get("last_name")
        username = user.get("username")
        display_name = self._display_name_from_parts(first_name, last_name, username)

        board = self.repo.get_board(chat_id)
        existing_owner = next((m for m in board.members.values() if m.is_owner), None)

        member = board.get_member(user_id)
        if member is None:
            if existing_owner is None:
                member = board.upsert_member(
                    user_id=user_id,
                    display_name=display_name,
                    role=ROLE_OWNER,
                    is_owner=True,
                )
            else:
                member = board.upsert_member(
                    user_id=user_id,
                    display_name=display_name,
                    role=ROLE_VIEWER,
                    is_owner=False,
                )
        else:
            if display_name:
                board.upsert_member(user_id=user_id, display_name=display_name)

        self.repo.save_board(board)
        return MemberWithRole(
            user_id=member.user_id,
            display_name=member.display_name,
            role=member.role,
            is_owner=member.is_owner,
        )

    def get_member(self, chat_id: int, user_id: int) -> Optional[MemberWithRole]:
        board = self.repo.get_board(chat_id)
        m = board.get_member(user_id)
        if not m:
            return None
        return MemberWithRole(
            user_id=m.user_id,
            display_name=m.display_name,
            role=m.role,
            is_owner=m.is_owner,
        )

    def list_members(self, chat_id: int) -> List[MemberWithRole]:
        board = self.repo.get_board(chat_id)
        out: List[MemberWithRole] = []
        for m in board.list_members():
            out.append(
                MemberWithRole(
                    user_id=m.user_id,
                    display_name=m.display_name,
                    role=m.role,
                    is_owner=m.is_owner,
                )
            )
        return out

    def remove_member(self, chat_id: int, user_id: int) -> None:
        """حذف عضو از برد و ذخیره‌سازی در مخزن (زمان خروج از گروه)."""
        board = self.repo.get_board(chat_id)
        board.remove_member(user_id)
        self.repo.save_board(board)

    def set_role(self, chat_id: int, actor_id: int, target_user_id: int,
                 new_role: str) -> MemberWithRole:
        """تنها owner می‌تواند نقش‌ها را تغییر دهد. owner قابل تغییر به نقش دیگر نیست."""
        if new_role not in ALL_ROLES:
            raise ValueError("نقش نامعتبر است.")

        board = self.repo.get_board(chat_id)
        actor = board.get_member(actor_id)
        if actor is None or not actor.is_owner:
            raise PermissionError("فقط owner می‌تواند نقش‌ها را مدیریت کند.")

        target = board.get_member(target_user_id)
        if target is None:
            raise KeyError("کاربر در اعضای برد ثبت نشده است.")
        if target.is_owner and new_role != ROLE_OWNER:
            raise PermissionError("نقش owner قابل تغییر به نقش دیگر نیست.")

        board.upsert_member(target_user_id, target.display_name, role=new_role,
                            is_owner=(new_role == ROLE_OWNER))
        self.repo.save_board(board)
        updated = board.get_member(target_user_id)
        return MemberWithRole(
            user_id=updated.user_id,
            display_name=updated.display_name,
            role=updated.role,
            is_owner=updated.is_owner,
        )

    def member_has_any_role(self, chat_id: int, user_id: int,
                            allowed_roles: List[str]) -> Tuple[bool, Optional[str]]:
        """بررسی این‌که آیا عضو یکی از نقش‌های مجاز (یا owner) را دارد یا نه."""
        m = self.get_member(chat_id, user_id)
        if not m:
            return False, None
        if m.is_owner:
            return True, ROLE_OWNER
        if m.role in allowed_roles:
            return True, m.role
        return False, m.role
