from __future__ import annotations

from dataclasses import dataclass, field
from datetime import datetime, date, time as dtime
from typing import Dict, List, Optional, Any


@dataclass
class Column:
    key: str
    name: str
    order_index: int
    wip_limit: Optional[int] = None
    is_done_column: bool = False


@dataclass
class Task:
    id: int
    title: str
    column_key: str
    creator_id: int
    assignee_id: Optional[int] = None
    created_at: datetime = field(default_factory=datetime.utcnow)
    done_at: Optional[datetime] = None
    class_of_service: str = "standard"
    due_at: Optional[datetime] = None
    is_blocked: bool = False
    blocked_reason: Optional[str] = None


@dataclass
class TaskEvent:
    task_id: int
    from_column: Optional[str]
    to_column: Optional[str]
    actor_id: Optional[int]
    created_at: datetime
    event_type: str
    extra: Dict[str, Any] = field(default_factory=dict)


@dataclass
class StandupConfig:
    team_chat_id: int
    enabled: bool = False
    time_of_day: Optional[dtime] = None
    questions: List[str] = field(default_factory=lambda: [
        "دیروز چه کردی؟",
        "امروز چه می‌کنی؟",
        "چه مانعی داری؟",
    ])


@dataclass
class StandupEntry:
    team_chat_id: int
    user_id: int
    standup_date: date
    answers: List[str]
    created_at: datetime = field(default_factory=datetime.utcnow)


@dataclass
class BoardMember:
    """عضو برد و نقش اسکرامبان در این گروه/چت."""

    user_id: int
    display_name: str
    role: str = "viewer"  # owner, po, sm, dev, viewer
    is_owner: bool = False


@dataclass
class Board:
    chat_id: int
    columns: List[Column] = field(default_factory=list)
    tasks: Dict[int, Task] = field(default_factory=dict)
    events: List[TaskEvent] = field(default_factory=list)
    next_task_id: int = 1
    standup_config: Optional[StandupConfig] = None
    standup_entries: List[StandupEntry] = field(default_factory=list)
    members: Dict[int, BoardMember] = field(default_factory=dict)

    # -------- ستون‌ها --------

    def ensure_default_columns(self) -> None:
        if self.columns:
            return
        self.columns = [
            Column(key="backlog", name="Backlog", order_index=1),
            Column(key="ready", name="Ready", order_index=2),
            Column(key="doing", name="Doing", order_index=3, wip_limit=3),
            Column(key="done", name="Done", order_index=4, is_done_column=True),
        ]

    def add_column(self, key: str, name: str, order_index: int,
                   wip_limit: Optional[int] = None,
                   is_done_column: bool = False) -> Column:
        if any(c.key == key for c in self.columns):
            raise ValueError("ستونی با این key از قبل وجود دارد.")
        col = Column(key=key, name=name, order_index=order_index,
                     wip_limit=wip_limit, is_done_column=is_done_column)
        self.columns.append(col)
        self.columns.sort(key=lambda c: c.order_index)
        return col

    def set_wip_limit(self, key: str, wip_limit: Optional[int]) -> None:
        for c in self.columns:
            if c.key == key:
                c.wip_limit = wip_limit
                return
        raise KeyError("ستون پیدا نشد.")

    def _get_column(self, key: str) -> Column:
        for c in self.columns:
            if c.key == key:
                return c
        raise KeyError("ستون پیدا نشد.")

    # -------- تسک‌ها --------

    def add_task(self, title: str, creator_id: int) -> Task:
        self.ensure_default_columns()
        task_id = self.next_task_id
        self.next_task_id += 1
        t = Task(
            id=task_id,
            title=title,
            column_key="backlog",
            creator_id=creator_id,
        )
        self.tasks[task_id] = t
        self.events.append(TaskEvent(
            task_id=task_id,
            from_column=None,
            to_column="backlog",
            actor_id=creator_id,
            created_at=datetime.utcnow(),
            event_type="create",
            extra={},
        ))
        return t

    def tasks_by_column(self, column_key: str) -> List[Task]:
        return [t for t in self.tasks.values() if t.column_key == column_key]

    def tasks_for_member(self, member_id: int) -> List[Task]:
        return [t for t in self.tasks.values() if t.assignee_id == member_id]

    def blocked_tasks(self) -> List[Task]:
        return [t for t in self.tasks.values() if t.is_blocked]

    def move_task(self, task_id: int, new_column_key: str,
                  actor_id: Optional[int] = None) -> Task:
        if task_id not in self.tasks:
            raise KeyError("Task not found")
        self.ensure_default_columns()
        self._get_column(new_column_key)
        task = self.tasks[task_id]
        from_col = task.column_key
        if new_column_key == "doing":
            doing_col = self._get_column("doing")
            if doing_col.wip_limit is not None:
                doing_count = len(self.tasks_by_column("doing"))
                if doing_count >= doing_col.wip_limit:
                    raise ValueError("WIP ستون Doing پر است.")
        task.column_key = new_column_key
        if new_column_key == "done":
            task.done_at = datetime.utcnow()
        self.events.append(TaskEvent(
            task_id=task_id,
            from_column=from_col,
            to_column=new_column_key,
            actor_id=actor_id,
            created_at=datetime.utcnow(),
            event_type="move",
            extra={},
        ))
        return task

    def block_task(self, task_id: int, reason: str,
                   actor_id: Optional[int] = None) -> Task:
        if task_id not in self.tasks:
            raise KeyError("Task not found")
        t = self.tasks[task_id]
        t.is_blocked = True
        t.blocked_reason = reason
        self.events.append(TaskEvent(
            task_id=task_id,
            from_column=t.column_key,
            to_column=t.column_key,
            actor_id=actor_id,
            created_at=datetime.utcnow(),
            event_type="block",
            extra={"reason": reason},
        ))
        return t

    def unblock_task(self, task_id: int,
                     actor_id: Optional[int] = None) -> Task:
        if task_id not in self.tasks:
            raise KeyError("Task not found")
        t = self.tasks[task_id]
        t.is_blocked = False
        t.blocked_reason = None
        self.events.append(TaskEvent(
            task_id=task_id,
            from_column=t.column_key,
            to_column=t.column_key,
            actor_id=actor_id,
            created_at=datetime.utcnow(),
            event_type="unblock",
            extra={},
        ))
        return t

    # -------- Standup --------

    def get_or_create_standup_config(self) -> StandupConfig:
        if self.standup_config is None:
            self.standup_config = StandupConfig(team_chat_id=self.chat_id)
        return self.standup_config

    def add_standup_entry(self, user_id: int, answers: List[str],
                           when: datetime | None = None) -> StandupEntry:
        when = when or datetime.utcnow()
        entry = StandupEntry(
            team_chat_id=self.chat_id,
            user_id=user_id,
            standup_date=when.date(),
            answers=answers,
            created_at=when,
        )
        self.standup_entries.append(entry)
        return entry

    # -------- اعضا و نقش‌ها --------

    def get_member(self, user_id: int) -> Optional[BoardMember]:
        return self.members.get(user_id)

    def upsert_member(self, user_id: int, display_name: str,
                      role: Optional[str] = None,
                      is_owner: Optional[bool] = None) -> BoardMember:
        m = self.members.get(user_id)
        if m is None:
            m = BoardMember(user_id=user_id, display_name=display_name)
            self.members[user_id] = m
        else:
            if display_name:
                m.display_name = display_name
        if role is not None:
            m.role = role
        if is_owner is not None:
            m.is_owner = is_owner
        return m

    def list_members(self) -> List[BoardMember]:
        return list(self.members.values())

    def remove_member(self, user_id: int) -> None:
        """حذف عضو از برد (مثلاً وقتی از گروه خارج می‌شود)."""
        self.members.pop(user_id, None)

