import logging
from typing import Any, Dict, List, Optional

import requests

from config.settings import get_settings

logger = logging.getLogger(__name__)


class BaleAPI:
    """Minimal HTTP client for Bale Bot API (getUpdates + sendMessage)."""

    def __init__(self) -> None:
        self.settings = get_settings()

    def _request(self, method: str, path: str, **kwargs) -> Dict[str, Any]:
        url = f"{self.settings.bale_base_url}/{path}"
        resp = requests.request(method, url, timeout=30, **kwargs)
        resp.raise_for_status()
        data = resp.json()
        if not data.get("ok", True):
            logger.error("Bale API error on %s: %s", path, data)
        return data

    def get_updates(self, offset: Optional[int] = None, timeout: Optional[int] = None) -> List[Dict[str, Any]]:
        params: Dict[str, Any] = {}
        if offset is not None:
            params["offset"] = offset
        params["timeout"] = timeout or self.settings.long_poll_timeout
        data = self._request("GET", "getUpdates", params=params)
        return data.get("result", [])

    def send_message(self, chat_id: int | str, text: str, reply_to_message_id: int | None = None) -> None:
        payload: Dict[str, Any] = {"chat_id": chat_id, "text": text}
        if reply_to_message_id is not None:
            payload["reply_to_message_id"] = reply_to_message_id
        self._request("POST", "sendMessage", json=payload)
