from __future__ import annotations

from dataclasses import dataclass, field
from datetime import datetime
from typing import Dict, List, Optional


@dataclass
class Column:
    key: str
    name: str
    order_index: int
    wip_limit: Optional[int] = None


@dataclass
class Task:
    id: int
    title: str
    column_key: str
    creator_id: int
    assignee_id: Optional[int] = None
    created_at: datetime = field(default_factory=datetime.utcnow)
    done_at: Optional[datetime] = None


@dataclass
class Board:
    chat_id: int
    columns: List[Column] = field(default_factory=list)
    tasks: Dict[int, Task] = field(default_factory=dict)
    next_task_id: int = 1

    def ensure_default_columns(self) -> None:
        if self.columns:
            return
        self.columns = [
            Column(key="backlog", name="Backlog", order_index=0),
            Column(key="ready", name="Ready", order_index=1),
            Column(key="doing", name="Doing", order_index=2, wip_limit=3),
            Column(key="done", name="Done", order_index=3),
        ]

    def get_column(self, key: str) -> Column:
        for c in self.columns:
            if c.key == key:
                return c
        raise KeyError(f"Column {key} not found")

    def add_task(self, title: str, creator_id: int) -> Task:
        self.ensure_default_columns()
        task = Task(
            id=self.next_task_id,
            title=title,
            column_key="backlog",
            creator_id=creator_id,
        )
        self.tasks[task.id] = task
        self.next_task_id += 1
        return task

    def move_task(self, task_id: int, new_column_key: str) -> Task:
        self.ensure_default_columns()
        if task_id not in self.tasks:
            raise KeyError("Task not found")
        task = self.tasks[task_id]
        if new_column_key == "doing":
            doing_wip = sum(1 for t in self.tasks.values() if t.column_key == "doing")
            doing_column = self.get_column("doing")
            if doing_column.wip_limit is not None and doing_wip >= doing_column.wip_limit:
                raise ValueError("حد WIP ستون Doing پر شده است.")
        task.column_key = new_column_key
        if new_column_key == "done":
            task.done_at = datetime.utcnow()
        return task

    def tasks_by_column(self, column_key: str):
        return [t for t in self.tasks.values() if t.column_key == column_key]

    def tasks_for_member(self, member_id: int):
        return [t for t in self.tasks.values() if t.assignee_id == member_id]
