from __future__ import annotations

import logging
from typing import Any, Dict, Optional

from bot.services.bale_api import BaleAPI
from bot.services.tasks_service import TasksService
from utils.validators import parse_int

logger = logging.getLogger(__name__)


class CommandHandler:
    """Routes Bale updates to command handlers.

    Supported commands in this MVP:
    - /backlog_new <title>   → add task to Backlog
    - /backlog               → list Backlog tasks
    - /to_ready <id>         → move Backlog → Ready
    - /board                 → show board
    - /start_task <id>       → move task to Doing
    - /done_task <id>        → move task to Done
    - /pull                  → pull first Ready task for member
    - /mytasks               → list tasks assigned to member
    """

    def __init__(self, api: BaleAPI | None = None, tasks_service: TasksService | None = None) -> None:
        self.api = api or BaleAPI()
        self.tasks = tasks_service or TasksService()

    def handle_update(self, update: Dict[str, Any]) -> None:
        message = update.get("message")
        if not message:
            return

        chat = message.get("chat") or {}
        chat_id = chat.get("id")
        if chat_id is None:
            return

        text: Optional[str] = message.get("text")
        if not text or not text.startswith("/"):
            # Non-command: ignore for now
            return

        from_user = message.get("from") or {}
        user_id = from_user.get("id")

        parts = text.strip().split(maxsplit=1)
        command = parts[0]
        arg = parts[1] if len(parts) > 1 else ""

        try:
            if command == "/backlog_new":
                self._cmd_backlog_new(chat_id, user_id, message, arg)
            elif command == "/backlog":
                self._cmd_backlog(chat_id, message)
            elif command == "/to_ready":
                self._cmd_to_ready(chat_id, message, arg)
            elif command == "/board":
                self._cmd_board(chat_id, message)
            elif command == "/start_task":
                self._cmd_move(chat_id, message, arg, new_column_key="doing")
            elif command == "/done_task":
                self._cmd_move(chat_id, message, arg, new_column_key="done")
            elif command == "/pull":
                self._cmd_pull(chat_id, user_id, message)
            elif command == "/mytasks":
                self._cmd_mytasks(chat_id, user_id, message)
            else:
                self.api.send_message(chat_id, "❓ دستور ناشناخته.")
        except Exception as exc:  # noqa: BLE001
            logger.exception("Error while handling command %s: %s", command, exc)
            self.api.send_message(chat_id, "❌ خطایی رخ داد. بعداً دوباره تلاش کنید.")

    # --- Command implementations ---

    def _cmd_backlog_new(self, chat_id: int, user_id: int, message: Dict[str, Any], arg: str) -> None:
        if not arg:
            self.api.send_message(
                chat_id,
                "برای تعریف کار جدید این‌گونه بنویس:\n"
                "/backlog_new عنوان کار",
                reply_to_message_id=message.get("message_id"),
            )
            return
        task = self.tasks.create_task(chat_id=chat_id, title=arg, creator_id=user_id)
        self.api.send_message(
            chat_id,
            f"✅ کار #{task.id} به Backlog اضافه شد.\nعنوان: {task.title}",
            reply_to_message_id=message.get("message_id"),
        )

    def _cmd_backlog(self, chat_id: int, message: Dict[str, Any]) -> None:
        tasks = self.tasks.list_tasks_by_column(chat_id, "backlog")
        if not tasks:
            text = "📥 Backlog خالی است."
        else:
            lines = ["📥 Backlog:"]
            for t in tasks:
                lines.append(f"- #{t.id} {t.title}")
            text = "\n".join(lines)
        self.api.send_message(chat_id, text, reply_to_message_id=message.get("message_id"))

    def _cmd_to_ready(self, chat_id: int, message: Dict[str, Any], arg: str) -> None:
        if not arg:
            self.api.send_message(
                chat_id,
                "استفاده:\n/to_ready <id>",
                reply_to_message_id=message.get("message_id"),
            )
            return
        ok, task_id = parse_int(arg)
        if not ok or task_id is None:
            self.api.send_message(
                chat_id,
                "❌ شناسهٔ کار باید عدد باشد.",
                reply_to_message_id=message.get("message_id"),
            )
            return
        try:
            task = self.tasks.move_task(chat_id, task_id, new_column_key="ready")
        except KeyError:
            self.api.send_message(
                chat_id,
                "❌ کار با این شناسه پیدا نشد.",
                reply_to_message_id=message.get("message_id"),
            )
            return

        self.api.send_message(
            chat_id,
            f"📌 کار #{task.id} به ستون Ready منتقل شد.",
            reply_to_message_id=message.get("message_id"),
        )

    def _cmd_board(self, chat_id: int, message: Dict[str, Any]) -> None:
        board = self.tasks.get_board(chat_id)
        lines: list[str] = ["📋 وضعیت فعلی برد:\n"]
        for col in sorted(board.columns, key=lambda c: c.order_index):
            tasks = [t for t in board.tasks.values() if t.column_key == col.key]
            header = f"{col.name}"
            if col.key == "doing" and col.wip_limit is not None:
                header += f" (WIP {len(tasks)}/{col.wip_limit})"
            lines.append(header + ":")
            if not tasks:
                lines.append("  - (خالی)")
            else:
                for t in tasks:
                    owner = f" (user {t.assignee_id})" if t.assignee_id is not None else ""
                    lines.append(f"  - #{t.id} {t.title}{owner}")
            lines.append("")
        text = "\n".join(lines)
        self.api.send_message(chat_id, text, reply_to_message_id=message.get("message_id"))

    def _cmd_move(self, chat_id: int, message: Dict[str, Any], arg: str, new_column_key: str) -> None:
        if not arg:
            usage = "/start_task <id>" if new_column_key == "doing" else "/done_task <id>"
            self.api.send_message(
                chat_id,
                f"استفاده:\n{usage}",
                reply_to_message_id=message.get("message_id"),
            )
            return
        ok, task_id = parse_int(arg)
        if not ok or task_id is None:
            self.api.send_message(
                chat_id,
                "❌ شناسهٔ کار باید عدد باشد.",
                reply_to_message_id=message.get("message_id"),
            )
            return

        try:
            task = self.tasks.move_task(chat_id, task_id, new_column_key=new_column_key)
        except ValueError as ve:
            self.api.send_message(
                chat_id,
                f"⚠️ {ve}",
                reply_to_message_id=message.get("message_id"),
            )
            return
        except KeyError:
            self.api.send_message(
                chat_id,
                "❌ کار با این شناسه پیدا نشد.",
                reply_to_message_id=message.get("message_id"),
            )
            return

        if new_column_key == "doing":
            msg = f"▶️ کار #{task.id} به Doing منتقل شد."
        elif new_column_key == "done":
            msg = f"🏁 کار #{task.id} به Done منتقل شد."
        else:
            msg = f"کار #{task.id} منتقل شد."

        self.api.send_message(chat_id, msg, reply_to_message_id=message.get("message_id"))

    def _cmd_pull(self, chat_id: int, user_id: Optional[int], message: Dict[str, Any]) -> None:
        if user_id is None:
            self.api.send_message(
                chat_id,
                "❌ شناسهٔ کاربر نامعتبر است.",
                reply_to_message_id=message.get("message_id"),
            )
            return
        try:
            task = self.tasks.pull_task_for_member(chat_id, user_id)
        except LookupError as le:
            self.api.send_message(
                chat_id,
                f"ℹ️ {le}",
                reply_to_message_id=message.get("message_id"),
            )
            return
        except ValueError as ve:
            self.api.send_message(
                chat_id,
                f"⚠️ {ve}",
                reply_to_message_id=message.get("message_id"),
            )
            return

        self.api.send_message(
            chat_id,
            f"✅ کار #{task.id} از Ready به Doing کشیده شد و به شما اختصاص یافت.",
            reply_to_message_id=message.get("message_id"),
        )

    def _cmd_mytasks(self, chat_id: int, user_id: Optional[int], message: Dict[str, Any]) -> None:
        if user_id is None:
            self.api.send_message(
                chat_id,
                "❌ شناسهٔ کاربر نامعتبر است.",
                reply_to_message_id=message.get("message_id"),
            )
            return
        tasks = self.tasks.list_tasks_for_member(chat_id, user_id)
        if not tasks:
            text = "📌 در حال حاضر هیچ کاری به شما اختصاص نیافته است."
        else:
            lines = ["📌 کارهای اختصاص‌یافته به شما:"]
            for t in tasks:
                lines.append(f"- #{t.id} [{t.column_key}] {t.title}")
            text = "\n".join(lines)
        self.api.send_message(chat_id, text, reply_to_message_id=message.get("message_id"))
