# -*- coding: utf-8 -*-
# membership_guard.py (refined)
# all comments are in Finglish; ui texts are Persian (formal)

import os
import asyncio
from typing import Tuple, Optional, Callable, Union
from telegram import Update, InlineKeyboardMarkup, InlineKeyboardButton, ReplyKeyboardMarkup
from telegram.ext import ContextTypes, ConversationHandler
from telegram.error import Forbidden

ChannelId = Union[int, str]

def _resolve_channel_id() -> ChannelId:
    """
    Accept CHANNEL_ID / REQUIRED_CHANNEL_ID env values.
    Support either numeric chat ids or @channel usernames.
    """
    raw = (
        os.getenv("CHANNEL_ID")
        or os.getenv("REQUIRED_CHANNEL_ID")
        or os.getenv("CHANNEL_CHAT_ID")
        or ""
    ).strip()
    if not raw:
        return -1003214490930
    if raw.startswith("@"):
        return raw
    try:
        value = int(raw)
    except ValueError:
        return raw
    if value > 0:
        value = -abs(value)
    return value

def _resolve_channel_url() -> str:
    """
    Prefer explicit INVITE_LINK/CHANNEL_PUBLIC_URL, then @username fallback.
    """
    for cand in (
        os.getenv("INVITE_LINK"),
        os.getenv("CHANNEL_PUBLIC_URL"),
        os.getenv("CHANNEL_URL"),
    ):
        if cand:
            return cand
    username = os.getenv("CHANNEL_USERNAME", "").lstrip("@")
    if username:
        return f"https://t.me/{username}"
    return "https://t.me/qverifyChnl"

def _env_flag(name: str, default: bool) -> bool:
    raw = os.getenv(name)
    if raw is None:
        return default
    return raw.strip().lower() in ("1", "true", "yes", "on")

# --- channel config ---
CHANNEL_ID: ChannelId = _resolve_channel_id()
CHANNEL_PUBLIC_URL: str = _resolve_channel_url()
CHANNEL_MEMBERSHIP_REQUIRED: bool = _env_flag("REQUIRE_CHANNEL_MEMBERSHIP", False)

# --- inline keyboard: join / recheck ---
JOIN_KB = InlineKeyboardMarkup([
    [InlineKeyboardButton("🔗 عضویت در کانال", url=CHANNEL_PUBLIC_URL)],
    [InlineKeyboardButton("عضو شدم ✅", callback_data="recheck_join")],
])

# --- fallback reply menus (used if main file is not imported) ---
# keep labels consistent and formal; same as main bot file
MAIN_MENU_USER = ReplyKeyboardMarkup(
    [["ثبت گارانتی جدید", "گارانتی‌های من"],
     ["کد فروشگاه", "کاتالوگ کالا"]],
    resize_keyboard=True, is_persistent=True
)

MAIN_MENU_ADMIN = ReplyKeyboardMarkup(
    [["ثبت گارانتی جدید", "گارانتی‌های من"],
     ["جست‌وجو", "لیست آخرین‌ها"],
     ["کد فروشگاه", "کاتالوگ کالا"],
     ["مدیریت فروشگاه ها", "مدیریت کاتالوگ ها"],
     ["خروجی اکسل", "راهنما"]],
    resize_keyboard=True, is_persistent=True
)

async def is_channel_member(bot, user_id: int) -> Tuple[bool, Optional[str]]:
    """
    return: (is_member, reason)
      reason = None             -> user is a member
      reason = "not_member"     -> user is not a member
      reason = "bot_not_admin"  -> bot is not admin and cannot check
    """
    try:
        mem = await bot.get_chat_member(chat_id=CHANNEL_ID, user_id=user_id)
        st = mem.status
        return (st in ("member", "creator", "administrator", "restricted"), None)
    except Forbidden:
        return (False, "bot_not_admin")
    except Exception:
        return (False, "not_member")

def require_membership(is_admin_fn: Callable[[int], bool], admin_only: bool = False):
    """
    decorator: check channel membership (and admin permission if needed)
    before running a handler
    """
    def deco(func):
        async def wrapped(update: Update, context: ContextTypes.DEFAULT_TYPE, *args, **kwargs):
            user = update.effective_user
            chat = update.effective_chat
            if not user or not chat:
                return

            if CHANNEL_MEMBERSHIP_REQUIRED:
                ok, reason = await is_channel_member(context.bot, user.id)
                if not ok:
                    msg = "برای استفاده از ربات، ابتدا عضو کانال شوید و سپس «عضو شدم ✅» را انتخاب کنید."
                    if reason == "bot_not_admin":
                        msg = "برای بررسی عضویت کاربران، لازم است ابتدا ربات را در کانال خود به‌عنوان ادمین اضافه کنید."
                    try:
                        if update.callback_query:
                            await update.callback_query.answer()
                            await update.callback_query.message.reply_text(msg, reply_markup=JOIN_KB)
                        else:
                            await context.bot.send_message(chat_id=chat.id, text=msg, reply_markup=JOIN_KB)
                    except Exception:
                        pass
                    return ConversationHandler.END

            if admin_only and not is_admin_fn(user.id):
                try:
                    await context.bot.send_message(
                        chat_id=chat.id,
                        text="دسترسی شما به این بخش محدود است.",
                        reply_markup=MAIN_MENU_USER
                    )
                except Exception:
                    pass
                return ConversationHandler.END

            return await func(update, context, *args, **kwargs)
        return wrapped
    return deco

async def on_recheck_join(update: Update, context: ContextTypes.DEFAULT_TYPE, is_admin_fn: Callable[[int], bool]):
    """handler for 'I've joined ✅' inline button; re-check membership without /start"""
    if not CHANNEL_MEMBERSHIP_REQUIRED:
        return
    query = update.callback_query
    await query.answer()
    uid = query.from_user.id

    # small delay to let the join state settle
    await asyncio.sleep(1.2)

    ok, reason = await is_channel_member(context.bot, uid)
    if ok:
        # remove inline markup on the same message (keep reply keyboard)
        try:
            await query.edit_message_reply_markup(reply_markup=None)
        except Exception:
            pass
        kb = MAIN_MENU_ADMIN if is_admin_fn(uid) else MAIN_MENU_USER
        await context.bot.send_message(
            chat_id=uid,
            text="سپاسگزاریم 🙏 عضویت شما در کانال تأیید شد.\nلطفاً یکی از گزینه‌های زیر را انتخاب کنید.",
            reply_markup=kb
        )
    else:
        msg = "به‌نظر می‌رسد هنوز عضو کانال نیستید. لطفاً عضو شوید و سپس «عضو شدم ✅» را انتخاب کنید."
        if reason == "bot_not_admin":
            msg = "ربات هنوز ادمین کانال نیست؛ لطفاً ربات را به‌عنوان ادمین به کانال اضافه کنید."
        try:
            await query.edit_message_text(msg, reply_markup=JOIN_KB)
        except Exception:
            await context.bot.send_message(chat_id=uid, text=msg, reply_markup=JOIN_KB)
