from __future__ import annotations
from decimal import Decimal
from typing import Optional
from sqlalchemy.orm import Session
from sqlalchemy import select, desc
from app.models import Order, OrderItem, Product, ProductPriceHistory, Payment
from app.schemas.order import CartItemIn, CheckoutRequest


def _current_price(db: Session, product_id: str) -> tuple[Decimal, int]:
    stmt = (
        select(ProductPriceHistory)
        .where(ProductPriceHistory.product_id == product_id)
        .order_by(desc(ProductPriceHistory.effective_from))
        .limit(1)
    )
    history = db.scalars(stmt).first()
    if not history:
        raise ValueError("price history missing")
    return history.base_price, history.discount_percent


def _recalculate(order: Order) -> None:
    total = sum(float(item.final_price_snapshot) for item in order.items)
    order.total = total


def get_or_create_cart(db: Session, tenant_id: str, customer_user_id: Optional[str]) -> Order:
    stmt = select(Order).where(Order.tenant_id == tenant_id, Order.customer_user_id == customer_user_id, Order.status == "cart")
    order = db.scalars(stmt).first()
    if order:
        return order
    order = Order(tenant_id=tenant_id, customer_user_id=customer_user_id, status="cart", total=0)
    db.add(order)
    db.commit()
    db.refresh(order)
    return order


def add_item(db: Session, order: Order, payload: CartItemIn) -> Order:
    product = db.get(Product, payload.product_id)
    if not product or str(product.tenant_id) != str(order.tenant_id):
        raise ValueError("invalid product")
    base_price, discount = _current_price(db, str(product.id))
    unit_price = float(base_price) * (100 - discount) / 100
    item = OrderItem(
        tenant_id=order.tenant_id,
        order_id=order.id,
        product_id=product.id,
        qty=payload.qty,
        unit_price_snapshot=base_price,
        discount_percent_snapshot=discount,
        final_price_snapshot=unit_price * payload.qty,
        chosen_specs_json=payload.chosen_specs,
    )
    order.items.append(item)
    _recalculate(order)
    db.add(order)
    db.commit()
    db.refresh(order)
    return order


def remove_item(db: Session, order: Order, item_id: str) -> Order:
    item = next((i for i in order.items if str(i.id) == item_id), None)
    if not item:
        raise ValueError("item not found")
    order.items.remove(item)
    db.delete(item)
    _recalculate(order)
    db.add(order)
    db.commit()
    db.refresh(order)
    return order


def save_cart(db: Session, order: Order) -> Order:
    order.status = "saved"
    db.add(order)
    db.commit()
    db.refresh(order)
    return order


def cancel_cart(db: Session, order: Order) -> None:
    db.delete(order)
    db.commit()


def checkout(db: Session, order: Order, payload: CheckoutRequest) -> Order:
    if not order.items:
        raise ValueError("cart empty")
    order.status = "awaiting_payment"
    order.payment_method = payload.payment_method
    db.add(order)
    db.commit()
    db.refresh(order)
    return order


def _serialize_payment(payment: Payment) -> dict:
    return {
        "id": str(payment.id),
        "method": payment.method,
        "amount": float(payment.amount),
        "status": payment.status,
        "gateway_tx_id": payment.gateway_tx_id,
        "created_at": payment.created_at.isoformat() if payment.created_at else None,
    }


def serialize_order(order: Order) -> dict:
    return {
        "id": str(order.id),
        "status": order.status,
        "total": float(order.total),
        "payment_method": order.payment_method,
        "items": [
            {
                "id": str(item.id),
                "product_id": str(item.product_id),
                "qty": item.qty,
                "final_price_snapshot": float(item.final_price_snapshot),
                "unit_price_snapshot": float(item.unit_price_snapshot),
                "discount_percent_snapshot": item.discount_percent_snapshot,
            }
            for item in order.items
        ],
        "payments": [_serialize_payment(payment) for payment in order.payments],
    }
