from __future__ import annotations
from datetime import datetime
from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy.orm import Session
from app.api.dependencies import get_db, get_tenant_context, require_roles
from app.schemas.visit import (
    VisitStartRequest,
    VisitCompleteRequest,
    VisitOut,
    LikeCreate,
    LikeOut,
)
from app.services import store_visits


router = APIRouter(prefix="/visits", tags=["visits"])


def _serialize_visit(visit) -> VisitOut:
    return VisitOut(
        id=visit.id,
        session_token=visit.session_token,
        status=visit.status,
        started_at=visit.started_at,
        ended_at=visit.ended_at,
    )


def _serialize_like(like) -> LikeOut:
    return LikeOut(
        id=like.id,
        product_id=like.product_id,
        telegram_user_id=like.telegram_user_id,
        liked_at=like.liked_at,
        session_id=like.session_id,
    )


@router.post("/start", response_model=VisitOut)
def start_visit(
    payload: VisitStartRequest,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
):
    visit = store_visits.start_visit(db, str(tenant.id), payload.telegram_user_id)
    return _serialize_visit(visit)


@router.post("/{session_token}/complete", response_model=VisitOut)
def complete_visit(
    session_token: str,
    payload: VisitCompleteRequest,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
):
    visit = store_visits.get_visit_by_token(db, str(tenant.id), session_token)
    if not visit:
        raise HTTPException(status_code=404, detail="visit not found")
    visit = store_visits.end_visit(db, visit, status=payload.status)
    return _serialize_visit(visit)


@router.get("/{session_token}", response_model=VisitOut)
def get_visit(session_token: str, tenant=Depends(get_tenant_context), db: Session = Depends(get_db)):
    visit = store_visits.get_visit_by_token(db, str(tenant.id), session_token)
    if not visit:
        raise HTTPException(status_code=404, detail="visit not found")
    return _serialize_visit(visit)


@router.post("/likes", response_model=LikeOut)
def create_like(
    payload: LikeCreate,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
):
    session = None
    if payload.session_token:
        session = store_visits.get_visit_by_token(db, str(tenant.id), payload.session_token)
        if not session:
            raise HTTPException(status_code=404, detail="visit not found")
    try:
        like = store_visits.record_like(
            db,
            str(tenant.id),
            str(payload.product_id),
            payload.telegram_user_id,
            session=session,
        )
    except ValueError as exc:
        raise HTTPException(status_code=404, detail=str(exc)) from exc
    return _serialize_like(like)


@router.get("/{session_token}/likes", response_model=list[LikeOut])
def likes_by_session(
    session_token: str,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
):
    session = store_visits.get_visit_by_token(db, str(tenant.id), session_token)
    if not session:
        raise HTTPException(status_code=404, detail="visit not found")
    likes = store_visits.list_likes_for_session(db, session)
    return [_serialize_like(like) for like in likes]


@router.get("/likes", response_model=list[LikeOut])
def likes_between(
    telegram_user_id: int,
    since: datetime = Query(...),
    until: datetime = Query(...),
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
):
    likes = store_visits.list_likes_between(db, str(tenant.id), telegram_user_id, since, until)
    return [_serialize_like(like) for like in likes]


@router.get("/me/likes", response_model=list[LikeOut])
def my_likes(
    tenant=Depends(get_tenant_context),
    user=Depends(require_roles("store_admin", "store_staff", "customer")),
    db: Session = Depends(get_db),
):
    if not user.telegram_user_id:
        raise HTTPException(status_code=400, detail="telegram id missing")
    likes = store_visits.list_likes_for_user(db, str(tenant.id), user.telegram_user_id)
    return [_serialize_like(like) for like in likes]


@router.delete("/me/likes/{product_id}", response_model=dict)
def delete_my_like(
    product_id: str,
    tenant=Depends(get_tenant_context),
    user=Depends(require_roles("store_admin", "store_staff", "customer")),
    db: Session = Depends(get_db),
):
    if not user.telegram_user_id:
        raise HTTPException(status_code=400, detail="telegram id missing")
    deleted = store_visits.delete_like(db, str(tenant.id), user.telegram_user_id, product_id)
    if not deleted:
        raise HTTPException(status_code=404, detail="like not found")
    return {"removed": True}
