from alembic import op
import sqlalchemy as sa

from app.db.types import GUID


TABLE_KWARGS = {"mysql_engine": "InnoDB", "mysql_charset": "utf8mb4"}


revision = "0001_base"
down_revision = None
branch_labels = None
depends_on = None


def upgrade():
    if op.get_bind().dialect.name == "postgresql":
        op.execute("CREATE EXTENSION IF NOT EXISTS pgcrypto;")

    op.create_table(
        "tenant",
        sa.Column("id", GUID(), primary_key=True),
        sa.Column("code", sa.String(255), nullable=False, unique=True),
        sa.Column("title", sa.Text(), nullable=False),
        sa.Column("category", sa.Text()),
        sa.Column("address", sa.Text()),
        sa.Column("geo_location", sa.JSON()),
        sa.Column("work_hours", sa.Text()),
        sa.Column("subscription_tier", sa.Text(), server_default="base", nullable=False),
        sa.Column("status", sa.Text(), server_default="active", nullable=False),
        sa.Column("created_at", sa.TIMESTAMP(timezone=True), server_default=sa.text("CURRENT_TIMESTAMP"), nullable=False),
        sa.Column("updated_at", sa.TIMESTAMP(timezone=True), server_default=sa.text("CURRENT_TIMESTAMP"), nullable=False),
        sa.CheckConstraint("subscription_tier IN ('base','silver','gold')"),
        sa.CheckConstraint("status IN ('active','inactive')"),
        **TABLE_KWARGS,
    )

    op.create_table(
        "bot",
        sa.Column("id", GUID(), primary_key=True),
        sa.Column("tenant_id", GUID(), sa.ForeignKey("tenant.id", ondelete="CASCADE")),
        sa.Column("type", sa.Text(), nullable=False),
        sa.Column("token", sa.String(255), nullable=False, unique=True),
        sa.Column("username", sa.Text()),
        sa.Column("verified_at", sa.TIMESTAMP(timezone=True)),
        sa.Column("is_active", sa.Boolean(), server_default=sa.text("TRUE"), nullable=False),
        sa.Column("created_at", sa.TIMESTAMP(timezone=True), server_default=sa.text("CURRENT_TIMESTAMP"), nullable=False),
        sa.Column("updated_at", sa.TIMESTAMP(timezone=True), server_default=sa.text("CURRENT_TIMESTAMP"), nullable=False),
        sa.CheckConstraint("type IN ('hyper','store')"),
        **TABLE_KWARGS,
    )

    op.create_table(
        "app_user",
        sa.Column("id", GUID(), primary_key=True),
        sa.Column("tenant_id", GUID(), sa.ForeignKey("tenant.id", ondelete="CASCADE")),
        sa.Column("telegram_user_id", sa.BigInteger()),
        sa.Column("login_username", sa.String(255)),
        sa.Column("password_hash", sa.Text()),
        sa.Column("phone", sa.Text()),
        sa.Column(
            "role",
            sa.Enum("hyper_admin", "store_admin", "store_staff", "customer", name="user_role"),
            nullable=False,
        ),
        sa.Column(
            "status",
            sa.Enum("active", "disabled", name="user_status"),
            server_default="active",
        ),
        sa.Column("created_at", sa.TIMESTAMP(timezone=True), server_default=sa.text("CURRENT_TIMESTAMP"), nullable=False),
        sa.Column("updated_at", sa.TIMESTAMP(timezone=True), server_default=sa.text("CURRENT_TIMESTAMP"), nullable=False),
        sa.CheckConstraint("role IN ('hyper_admin','store_admin','store_staff','customer')"),
        sa.CheckConstraint("status IN ('active','disabled')"),
        **TABLE_KWARGS,
    )
    op.create_index("idx_app_user_tenant_role", "app_user", ["tenant_id", "role"])
    op.create_unique_constraint("uq_user_tenant_username", "app_user", ["tenant_id", "login_username"])

    op.create_table(
        "store_profile",
        sa.Column("tenant_id", GUID(), sa.ForeignKey("tenant.id", ondelete="CASCADE"), primary_key=True),
        sa.Column("about_text", sa.Text()),
        sa.Column("support_text", sa.Text()),
        sa.Column("contact_phone", sa.Text()),
        sa.Column("logo_file_id", sa.Text()),
        sa.Column("updated_at", sa.TIMESTAMP(timezone=True), server_default=sa.text("CURRENT_TIMESTAMP"), nullable=False),
        **TABLE_KWARGS,
    )

    op.create_table(
        "account_request",
        sa.Column("id", GUID(), primary_key=True),
        sa.Column("full_name", sa.Text()),
        sa.Column("phone", sa.Text()),
        sa.Column("description", sa.Text()),
        sa.Column("status", sa.Text(), server_default="new", nullable=False),
        sa.Column("created_at", sa.TIMESTAMP(timezone=True), server_default=sa.text("CURRENT_TIMESTAMP"), nullable=False),
        sa.CheckConstraint("status IN ('new','in_progress','done','rejected')"),
        **TABLE_KWARGS,
    )


def downgrade():
    op.drop_table("account_request")
    op.drop_table("store_profile")
    op.drop_constraint("uq_user_tenant_username", "app_user", type_="unique")
    op.drop_index("idx_app_user_tenant_role", table_name="app_user")
    op.drop_table("app_user")
    op.drop_table("bot")
    op.drop_table("tenant")
