from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from app.api.dependencies import get_db, get_tenant_context, require_roles
from app.services import users as user_service
from app.schemas.user import UserCreate, UserUpdate, UserOut
from app.models import AppUser


router = APIRouter(prefix="/users", tags=["users"])


@router.get("/", response_model=list[UserOut])
def list_users(
    role: str | None = None,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("hyper_admin", "store_admin")),
):
    records = user_service.list_users(db, str(tenant.id), role)
    return [UserOut(id=r.id, username=r.login_username or "", role=r.role, status=r.status) for r in records]


@router.post("/", response_model=UserOut)
def create_user(
    payload: UserCreate,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("hyper_admin", "store_admin")),
):
    user = user_service.create_user(db, str(tenant.id), payload)
    return UserOut(id=user.id, username=user.login_username or "", role=user.role, status=user.status)


@router.put("/{user_id}", response_model=UserOut)
def update_user(
    user_id: str,
    payload: UserUpdate,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("hyper_admin", "store_admin")),
):
    user = db.get(AppUser, user_id)
    if not user or str(user.tenant_id) != str(tenant.id):
        raise HTTPException(status_code=404, detail="user not found")
    user = user_service.update_user(db, user, payload)
    return UserOut(id=user.id, username=user.login_username or "", role=user.role, status=user.status)


@router.delete("/{user_id}")
def delete_user(
    user_id: str,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("hyper_admin", "store_admin")),
):
    user = db.get(AppUser, user_id)
    if not user or str(user.tenant_id) != str(tenant.id):
        raise HTTPException(status_code=404, detail="user not found")
    user_service.delete_user(db, user)
    return {"deleted": True}
