from __future__ import annotations
from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from sqlalchemy import select
from sqlalchemy.exc import IntegrityError
from app.api.dependencies import get_db, require_roles, get_tenant_context
from app.models import Tenant, StoreProfile
from app.schemas.tenant import (
    TenantCreate,
    TenantUpdate,
    TenantOut,
    TenantDetail,
    TenantAdminCreate,
    TenantAdminOut,
    StoreProfileIn,
    StoreProfileOut,
    StoreSettingsOut,
    TenantNotification,
)
from app.services.tenants import create_tenant, update_tenant, add_tenant_admin
from app.services import notifications


router = APIRouter(prefix="/tenants", tags=["tenants"])


def _normalize_manager_ids(raw: list[int | str] | None) -> list[int]:
    if not raw:
        return []
    normalized: list[int] = []
    for value in raw:
        try:
            num = int(value)
        except (TypeError, ValueError):
            continue
        if num > 0:
            normalized.append(num)
    return normalized


def _serialize_tenant(tenant: Tenant) -> TenantOut:
    return TenantOut(
        id=tenant.id,
        code=tenant.code,
        title=tenant.title,
        subscription_tier=tenant.subscription_tier,
        status=tenant.status,
        category=tenant.category,
        address=tenant.address,
    )


@router.get("/", response_model=list[TenantOut])
def list_tenants(
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("hyper_admin")),
):
    tenants = db.scalars(select(Tenant)).all()
    return [_serialize_tenant(tenant) for tenant in tenants]


@router.get("/{tenant_id}", response_model=TenantDetail)
def get_tenant(
    tenant_id: str,
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("hyper_admin")),
):
    tenant = db.get(Tenant, tenant_id)
    if not tenant:
        raise HTTPException(status_code=404, detail="tenant not found")
    profile = db.get(StoreProfile, tenant_id)
    return TenantDetail(
        **_serialize_tenant(tenant).dict(),
        contact_phone=profile.contact_phone if profile else None,
        work_hours=tenant.work_hours,
    )


@router.post("/", response_model=dict)
async def create_tenant_endpoint(
    payload: TenantCreate,
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("hyper_admin")),
):
    try:
        tenant, username, password = await create_tenant(db, payload)
    except ValueError as exc:
        raise HTTPException(status_code=400, detail=str(exc)) from exc
    except IntegrityError as exc:
        raise HTTPException(status_code=400, detail="bot token already registered") from exc
    return {
        "id": str(tenant.id),
        "code": tenant.code,
        "admin_credentials": {"username": username, "password": password},
    }


@router.put("/{tenant_id}", response_model=TenantOut)
def update_tenant_endpoint(
    tenant_id: str,
    payload: TenantUpdate,
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("hyper_admin")),
):
    tenant = db.get(Tenant, tenant_id)
    if not tenant:
        raise HTTPException(status_code=404, detail="tenant not found")
    tenant = update_tenant(db, tenant, payload)
    return _serialize_tenant(tenant)


@router.post("/{tenant_id}/admins", response_model=TenantAdminOut)
def add_admin(
    tenant_id: str,
    payload: TenantAdminCreate,
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("hyper_admin")),
):
    tenant = db.get(Tenant, tenant_id)
    if not tenant:
        raise HTTPException(status_code=404, detail="tenant not found")
    admin = add_tenant_admin(db, tenant, payload)
    return TenantAdminOut(id=admin.id, username=admin.login_username, role=admin.role, status=admin.status)


@router.put("/{tenant_id}/profile", response_model=dict)
def update_profile(
    tenant_id: str,
    payload: StoreProfileIn,
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("hyper_admin", "store_admin")),
):
    profile = db.get(StoreProfile, tenant_id)
    if not profile:
        profile = StoreProfile(tenant_id=tenant_id)
    data = payload.dict(exclude_unset=True)
    if "manager_telegram_ids" in data:
        if data["manager_telegram_ids"] is None:
            data["manager_telegram_ids"] = []
        else:
            data["manager_telegram_ids"] = _normalize_manager_ids(data["manager_telegram_ids"])
    for field, value in data.items():
        setattr(profile, field, value)
    db.add(profile)
    db.commit()
    return {"tenant_id": tenant_id, "updated": True}


@router.get("/{tenant_id}/profile", response_model=StoreProfileOut)
def get_profile(
    tenant_id: str,
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("hyper_admin", "store_admin")),
):
    profile = db.get(StoreProfile, tenant_id)
    if not profile:
        raise HTTPException(status_code=404, detail="profile not found")
    return StoreProfileOut(
        about_text=profile.about_text,
        support_text=profile.support_text,
        contact_phone=profile.contact_phone,
        logo_file_id=profile.logo_file_id,
        channel_id=profile.channel_id,
        channel_invite_link=profile.channel_invite_link,
        manager_telegram_ids=_normalize_manager_ids(profile.manager_telegram_ids),
        payment_link_template=profile.payment_link_template,
        payment_link_note=profile.payment_link_note,
        updated_at=profile.updated_at.isoformat() if profile.updated_at else None,
    )


@router.get("/current/settings", response_model=StoreSettingsOut)
def current_settings(
    tenant=Depends(get_tenant_context),
    _: object = Depends(require_roles("store_admin", "store_staff", "customer")),
    db: Session = Depends(get_db),
):
    profile = db.get(StoreProfile, tenant.id)
    return StoreSettingsOut(
        tenant_id=tenant.id,
        manager_telegram_ids=_normalize_manager_ids(profile.manager_telegram_ids if profile else None),
        payment_link_template=profile.payment_link_template if profile else None,
        payment_link_note=profile.payment_link_note if profile else None,
    )


@router.post("/{tenant_id}/notify", response_model=dict)
def notify_tenant(
    tenant_id: str,
    payload: TenantNotification,
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("hyper_admin")),
):
    tenant = db.get(Tenant, tenant_id)
    if not tenant:
        raise HTTPException(status_code=404, detail="tenant not found")
    notifications.notify_store_admin(tenant.code, payload.message)
    return {"sent": True}
