from __future__ import annotations

from typing import Any

from aiogram import Router, F, types
from aiogram.filters import Command
from aiogram.fsm.context import FSMContext
from aiogram.types import ReplyKeyboardMarkup, KeyboardButton, InlineKeyboardMarkup, InlineKeyboardButton, ReplyKeyboardRemove

from app.bots.hyper.states import LeadStates
from app.bots.hyper import keyboards
from app.bots.shared.api_client import BackendClient


router = Router()


def _location_keyboard() -> ReplyKeyboardMarkup:
    return ReplyKeyboardMarkup(
        keyboard=[[KeyboardButton(text="📍 ارسال موقعیت مکانی", request_location=True)]],
        resize_keyboard=True,
        one_time_keyboard=True,
    )


@router.message(F.text == "📮 درخواست خرید حساب")
async def account_purchase_start(message: types.Message, state: FSMContext):
    await state.set_state(LeadStates.store_title)
    await state.update_data(account_purchase={})
    await message.answer(
        "برای راه‌اندازی فروشگاه خودکار، ابتدا نام فروشگاه یا مارکت را ارسال کنید.",
        reply_markup=ReplyKeyboardRemove(),
    )


@router.message(LeadStates.store_title, Command("cancel"))
@router.message(LeadStates.owner_full_name, Command("cancel"))
@router.message(LeadStates.phone, Command("cancel"))
@router.message(LeadStates.address, Command("cancel"))
@router.message(LeadStates.business_description, Command("cancel"))
@router.message(LeadStates.location, Command("cancel"))
@router.message(LeadStates.bot_token, Command("cancel"))
@router.message(LeadStates.channel_id, Command("cancel"))
@router.message(LeadStates.channel_link, Command("cancel"))
async def account_purchase_cancel(message: types.Message, state: FSMContext):
    await state.clear()
    await message.answer("فرایند خرید حساب لغو شد.", reply_markup=keyboards.main_menu())


@router.message(LeadStates.store_title)
async def account_purchase_store_title(message: types.Message, state: FSMContext):
    text = (message.text or "").strip()
    if len(text) < 3:
        await message.answer("نام فروشگاه باید حداقل ۳ کاراکتر باشد. لطفاً دوباره ارسال کنید.")
        return
    data = await state.get_data()
    payload: dict[str, Any] = data.get("account_purchase", {})
    payload["store_title"] = text
    await state.update_data(account_purchase=payload)
    await state.set_state(LeadStates.owner_full_name)
    await message.answer("نام و نام خانوادگی مدیر فروشگاه را ارسال کنید.")


@router.message(LeadStates.owner_full_name)
async def account_purchase_owner_name(message: types.Message, state: FSMContext):
    text = (message.text or "").strip()
    if len(text) < 3:
        await message.answer("نام وارد‌شده معتبر نیست. لطفاً نام و نام خانوادگی را کامل وارد کنید.")
        return
    data = await state.get_data()
    payload: dict[str, Any] = data.get("account_purchase", {})
    payload["owner_full_name"] = text
    await state.update_data(account_purchase=payload)
    await state.set_state(LeadStates.phone)
    await message.answer("شماره موبایل مدیر را ارسال کنید (مثلاً 09123456789).")


@router.message(LeadStates.phone)
async def account_purchase_phone(message: types.Message, state: FSMContext):
    phone = (message.text or "").strip()
    if not phone:
        await message.answer("شماره موبایل خالی است. لطفاً دوباره ارسال کنید.")
        return
    data = await state.get_data()
    payload: dict[str, Any] = data.get("account_purchase", {})
    payload["phone"] = phone
    await state.update_data(account_purchase=payload)
    await state.set_state(LeadStates.address)
    await message.answer("آدرس فروشگاه را ارسال کنید.")


@router.message(LeadStates.address)
async def account_purchase_address(message: types.Message, state: FSMContext):
    text = (message.text or "").strip()
    if len(text) < 5:
        await message.answer("آدرس خیلی کوتاه است. لطفاً آدرس کامل‌تری ارسال کنید.")
        return
    data = await state.get_data()
    payload: dict[str, Any] = data.get("account_purchase", {})
    payload["address"] = text
    await state.update_data(account_purchase=payload)
    await state.set_state(LeadStates.business_description)
    await message.answer("توضیح کوتاهی درباره کسب‌وکار خود ارسال کنید (مثلاً نوع محصولات، شهر و ...).")


@router.message(LeadStates.business_description)
async def account_purchase_description(message: types.Message, state: FSMContext):
    text = (message.text or "").strip()
    data = await state.get_data()
    payload: dict[str, Any] = data.get("account_purchase", {})
    payload["business_description"] = text
    await state.update_data(account_purchase=payload)
    await state.set_state(LeadStates.location)
    await message.answer(
        "لوکیشن فروشگاه را به صورت Location تلگرام ارسال کنید.",
        reply_markup=_location_keyboard(),
    )


@router.message(LeadStates.location, F.location)
async def account_purchase_location(message: types.Message, state: FSMContext):
    loc = message.location
    data = await state.get_data()
    payload: dict[str, Any] = data.get("account_purchase", {})
    payload["lat"] = loc.latitude
    payload["lng"] = loc.longitude
    await state.update_data(account_purchase=payload)
    await state.set_state(LeadStates.bot_token)
    await message.answer(
        "توکن ربات فروشگاه (Bot Token) را ارسال کنید.",
        reply_markup=ReplyKeyboardRemove(),
    )


@router.message(LeadStates.location)
async def account_purchase_location_invalid(message: types.Message, state: FSMContext):
    await message.answer("برای این مرحله باید موقعیت مکانی را به صورت Location ارسال کنید.", reply_markup=_location_keyboard())


@router.message(LeadStates.bot_token)
async def account_purchase_bot_token(message: types.Message, state: FSMContext):
    token = (message.text or "").strip()
    if not token or ":" not in token:
        await message.answer("توکن ارسال‌شده معتبر نیست. لطفاً Bot Token کامل را ارسال کنید.")
        return
    data = await state.get_data()
    payload: dict[str, Any] = data.get("account_purchase", {})
    payload["bot_token"] = token
    await state.update_data(account_purchase=payload)
    await state.set_state(LeadStates.channel_id)
    await message.answer(
        "شناسه کانال فروشگاه را ارسال کنید (مثلاً @MyStoreChannel یا شناسه عددی)."
    )


@router.message(LeadStates.channel_id)
async def account_purchase_channel_id(message: types.Message, state: FSMContext):
    text = (message.text or "").strip()
    if not text:
        await message.answer("شناسه کانال خالی است. لطفاً دوباره ارسال کنید.")
        return
    data = await state.get_data()
    payload: dict[str, Any] = data.get("account_purchase", {})
    payload["channel_id"] = text
    await state.update_data(account_purchase=payload)
    await state.set_state(LeadStates.channel_link)
    await message.answer("لینک دعوت کانال فروشگاه را ارسال کنید (لینک join).")


@router.message(LeadStates.channel_link)
async def account_purchase_channel_link(message: types.Message, state: FSMContext):
    text = (message.text or "").strip()
    data = await state.get_data()
    payload: dict[str, Any] = data.get("account_purchase", {})
    payload["channel_link"] = text
    await state.update_data(account_purchase=payload)

    # خلاصه اطلاعات و دکمه تأیید پرداخت
    lines = [
        f"نام فروشگاه: {payload.get('store_title')}",
        f"مدیر: {payload.get('owner_full_name')}",
        f"موبایل مدیر: {payload.get('phone')}",
        f"آدرس: {payload.get('address')}",
        f"توضیح کسب‌وکار: {payload.get('business_description')}",
        f"Bot Token: {payload.get('bot_token')[:10]}...",
        f"کانال: {payload.get('channel_id')}",
        f"لینک کانال: {payload.get('channel_link')}",
    ]
    summary = "خلاصه اطلاعات ثبت‌شده:\n" + "\n".join(lines)
    kb = InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="✅ تأیید و رفتن به پرداخت", callback_data="acctbuy:confirm")],
            [InlineKeyboardButton(text="❌ لغو", callback_data="acctbuy:cancel")],
        ]
    )
    await message.answer(summary)
    await message.answer(
        "اگر اطلاعات بالا صحیح است، روی دکمه «تأیید و رفتن به پرداخت» بزنید.",
        reply_markup=kb,
    )


@router.callback_query(F.data == "acctbuy:cancel")
async def account_purchase_confirm_cancel(callback: types.CallbackQuery, state: FSMContext):
    await state.clear()
    await callback.answer()
    await callback.message.answer("فرایند خرید حساب لغو شد.", reply_markup=keyboards.main_menu())


@router.callback_query(F.data == "acctbuy:confirm")
async def account_purchase_confirm(callback: types.CallbackQuery, state: FSMContext):
    await callback.answer()
    data = await state.get_data()
    payload: dict[str, Any] = data.get("account_purchase") or {}

    required_keys = [
        "store_title",
        "owner_full_name",
        "phone",
        "address",
        "business_description",
        "lat",
        "lng",
        "bot_token",
        "channel_id",
        "channel_link",
    ]
    missing = [k for k in required_keys if not payload.get(k)]
    if missing:
        await callback.message.answer(
            "برخی از اطلاعات لازم کامل نیست. لطفاً فرایند را از ابتدا تکرار کنید.",
            reply_markup=keyboards.main_menu(),
        )
        await state.clear()
        return

    body = {
        "store_title": payload["store_title"],
        "owner_full_name": payload["owner_full_name"],
        "phone": payload["phone"],
        "address": payload["address"],
        "business_description": payload["business_description"],
        "lat": payload["lat"],
        "lng": payload["lng"],
        "bot_token": payload["bot_token"],
        "channel_id": payload["channel_id"],
        "channel_link": payload["channel_link"],
        "telegram_user_id": callback.from_user.id,
    }

    client = BackendClient()
    waiting = await callback.message.answer("در حال ایجاد لینک پرداخت...")
    try:
        resp = await client.post("/public/account-purchase/init", body)
    except Exception as exc:
        await waiting.edit_text(f"ایجاد پرداخت با خطا مواجه شد: {exc}")
        await state.clear()
        await callback.message.answer("بعداً دوباره تلاش کنید.", reply_markup=keyboards.main_menu())
        return

    payment_url = resp.get("payment_url")
    if not payment_url:
        await waiting.edit_text("خطا: در پاسخ درگاه پرداخت لینک معتبری دریافت نشد.")
        await state.clear()
        await callback.message.answer("بعداً دوباره تلاش کنید.", reply_markup=keyboards.main_menu())
        return

    pay_kb = InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="💳 پرداخت با زرین‌پال", url=payment_url)],
        ]
    )
    await waiting.edit_text(
        "لطفاً با دکمه زیر به صفحه زرین‌پال بروید و پرداخت را انجام دهید.\n"
        "پس از تکمیل پرداخت، طی چند دقیقه اطلاعات فروشگاه و ربات برای شما در همین گفتگو ارسال می‌شود.",
        reply_markup=pay_kb,
    )
    await state.clear()
