from sqlalchemy.orm import Session
from sqlalchemy import select
from app.models import AppUser
from app.schemas.user import UserCreate, UserUpdate
from app.core.security import hash_password
from app.utils.validators import ensure_iran_phone


def list_users(db: Session, tenant_id: str, role: str | None = None) -> list[AppUser]:
    stmt = select(AppUser).where(AppUser.tenant_id == tenant_id)
    if role:
        stmt = stmt.where(AppUser.role == role)
    return list(db.scalars(stmt))


def create_user(db: Session, tenant_id: str, payload: UserCreate) -> AppUser:
    if payload.phone:
        ensure_iran_phone(payload.phone)
    user = AppUser(
        tenant_id=tenant_id,
        login_username=payload.username,
        password_hash=hash_password(payload.password),
        role=payload.role,
        phone=payload.phone,
        status="active",
    )
    db.add(user)
    db.commit()
    db.refresh(user)
    return user


def update_user(db: Session, user: AppUser, payload: UserUpdate) -> AppUser:
    # username change
    if payload.username and payload.username != user.login_username:
        # ensure uniqueness inside tenant
        exists = db.scalars(
            select(AppUser).where(
                AppUser.tenant_id == user.tenant_id,
                AppUser.login_username == payload.username,
                AppUser.id != user.id,
            )
        ).first()
        if exists:
            raise ValueError("username already in use")
        user.login_username = payload.username

    if payload.password:
        user.password_hash = hash_password(payload.password)
    if payload.phone:
        ensure_iran_phone(payload.phone)
        user.phone = payload.phone
    if payload.status:
        user.status = payload.status
    db.add(user)
    db.commit()
    db.refresh(user)
    return user


def delete_user(db: Session, user: AppUser) -> None:
    db.delete(user)
    db.commit()
