from __future__ import annotations
import secrets
from datetime import datetime, timezone
from sqlalchemy.orm import Session
from sqlalchemy import select
from app.models import Tenant, Bot, AppUser, StoreProfile
from app.schemas.tenant import TenantCreate, TenantUpdate, TenantAdminCreate
from app.core.security import hash_password
from app.utils.validators import ensure_iran_phone
from app.core.logging import get_logger
from app.services.telegram import verify_bot_token


logger = get_logger(__name__)


def _generate_code(title: str) -> str:
    base = ''.join(filter(str.isalnum, title.lower()))[:8] or 'tenant'
    suffix = secrets.token_hex(2)
    return f"{base}-{suffix}"


async def create_tenant(db: Session, payload: TenantCreate) -> tuple[Tenant, str, str]:
    ensure_iran_phone(payload.store_admin_phone)
    if payload.code:
        existing = db.execute(select(Tenant).where(Tenant.code == payload.code)).scalar_one_or_none()
        if existing:
            raise ValueError("tenant code already exists")
        code = payload.code
    else:
        code = _generate_code(payload.title)
    username = payload.store_admin_username or f"{code}_admin"
    plain_password = payload.store_admin_password or secrets.token_urlsafe(8)
    password_hash = hash_password(plain_password)

    bot_info = await verify_bot_token(payload.bot_token)
    bot_username = bot_info.get("username")

    tenant = Tenant(
        code=code,
        title=payload.title,
        category=payload.category,
        address=payload.address,
        geo_location=payload.geo_location,
        work_hours=payload.work_hours,
        subscription_tier=payload.subscription_tier,
    )
    db.add(tenant)
    db.flush()

    bot = Bot(
        tenant_id=tenant.id,
        type="store",
        token=payload.bot_token,
        username=bot_username,
        verified_at=datetime.now(timezone.utc),
    )
    db.add(bot)

    admin_user = AppUser(
        tenant_id=tenant.id,
        login_username=username,
        password_hash=password_hash,
        role="store_admin",
        phone=payload.store_admin_phone,
        status="active",
    )
    db.add(admin_user)

    profile = StoreProfile(
        tenant_id=tenant.id,
        contact_phone=payload.contact_phone or payload.store_admin_phone,
        about_text=f"Store {tenant.title}",
        support_text="",
    )
    db.add(profile)

    db.commit()
    db.refresh(tenant)
    return tenant, username, plain_password


def update_tenant(db: Session, tenant: Tenant, payload: TenantUpdate) -> Tenant:
    for field, value in payload.dict(exclude_unset=True).items():
        if hasattr(tenant, field):
            setattr(tenant, field, value)
    db.add(tenant)
    db.commit()
    db.refresh(tenant)
    return tenant


def add_tenant_admin(db: Session, tenant: Tenant, payload: TenantAdminCreate) -> AppUser:
    ensure_iran_phone(payload.phone) if payload.phone else None
    user = AppUser(
        tenant_id=tenant.id,
        login_username=payload.username,
        password_hash=hash_password(payload.password),
        role="store_admin",
        phone=payload.phone,
        status="active",
    )
    db.add(user)
    db.commit()
    db.refresh(user)
    return user
