from __future__ import annotations
import asyncio
import re
import uuid
from typing import Any
from aiogram import Router, F, types
try:
    from aiogram.exceptions import TelegramBadRequest, TelegramForbidden
except ImportError:  # pragma: no cover - older aiogram versions
    from aiogram.exceptions import TelegramAPIError, TelegramBadRequest

    TelegramForbidden = TelegramAPIError
from aiogram.filters import CommandStart, Command
from aiogram.fsm.context import FSMContext
from aiogram.types import InlineKeyboardMarkup, InlineKeyboardButton
import httpx
from app.bots.store.states import (
    StoreLoginStates,
    AdminMenuStates,
    AdminFlowStates,
    ProductWizardStates,
    CategoryWizardStates,
    OrderStates,
    ChannelStates,
    AdminSettingsStates,
    CustomerFlowStates,
    CustomerAccountStates,
)
from app.bots.store import keyboards
from app.bots.shared.api_client import BackendClient
from app.core.config import get_settings
from contextvars import ContextVar
from app.core.logging import get_logger


router = Router()
logger = get_logger("app.bots.store")
settings = get_settings()

# Context var to inject tenant id for a running store bot process
STORE_TENANT_ID_CTX: ContextVar[str | None] = ContextVar("STORE_TENANT_ID_CTX", default=None)
CUSTOMER_SESSIONS: dict[int, dict[str, Any]] = {}
_TOKEN_UNSET = object()
ADMIN_ROLES = {"store_admin", "store_staff", "hyper_admin", "admin"}

PRODUCT_FLOW = [
    {"key": "title", "label": "عنوان محصول", "type": "text"},
    {"key": "description", "label": "توضیحات", "type": "text"},
    {"key": "base_price", "label": "قیمت (تومان)", "type": "number"},
    {"key": "discount_percent", "label": "درصد تخفیف", "type": "number"},
    {"key": "stock", "label": "موجودی", "type": "number"},
    {"key": "image_file_id", "label": "تصویر اصلی", "type": "photo"},
    {"key": "categories", "label": "دسته‌بندی‌ها", "type": "categories"},
]
PRODUCT_MAP = {step["key"]: step for step in PRODUCT_FLOW}


# ---------------------------------------------------------------------------
# Helpers
# ---------------------------------------------------------------------------


async def _get_client(state: FSMContext, token_key: str = "token") -> BackendClient:
    data = await state.get_data()
    if not data.get("user_role"):
        await state.update_data(user_role="customer")
        data = await state.get_data()
    token = data.get(token_key)
    if not token:
        raise RuntimeError("missing token")
    return BackendClient(token=token)


def _tenant_id_from_state(data: dict[str, Any]) -> str:
    tenant_id = data.get("tenant_id") or STORE_TENANT_ID_CTX.get() or settings.STORE_TENANT_ID
    if not tenant_id:
        raise RuntimeError("missing tenant id")
    return tenant_id


def _customer_tenant_id(data: dict[str, Any]) -> str | None:
    return data.get("cust_tenant_id") or STORE_TENANT_ID_CTX.get() or settings.STORE_TENANT_ID


def _session_customer_token(entry: dict[str, Any] | None) -> str | None:
    if not entry:
        return None
    token = entry.get("customer_token")
    if token:
        return token
    if _normalize_role(entry.get("role")) == "admin":
        return None
    return entry.get("token")


def _session_admin_token(entry: dict[str, Any] | None) -> str | None:
    if not entry:
        return None
    token = entry.get("admin_token")
    if token:
        return token
    if _normalize_role(entry.get("role")) == "admin":
        return entry.get("token")
    return None


def _session_update(user_id: int, token: Any = _TOKEN_UNSET, token_scope: str | None = None, **kwargs: Any) -> None:
    entry = CUSTOMER_SESSIONS.get(user_id, {})
    role = kwargs.get("role", entry.get("role"))
    if token is not _TOKEN_UNSET:
        scope = token_scope or ("admin" if _normalize_role(role) == "admin" else "customer")
        if token:
            if scope in {"customer", "all"}:
                entry["customer_token"] = token
                entry["token"] = token
            if scope in {"admin", "all"}:
                entry["admin_token"] = token
        else:
            if scope in {"customer", "all"}:
                entry.pop("customer_token", None)
                entry.pop("token", None)
            if scope in {"admin", "all"}:
                entry.pop("admin_token", None)
    for key, value in kwargs.items():
        if value is None:
            entry.pop(key, None)
        else:
            entry[key] = value
    if entry:
        CUSTOMER_SESSIONS[user_id] = entry
    elif user_id in CUSTOMER_SESSIONS:
        CUSTOMER_SESSIONS.pop(user_id, None)


def _normalize_role(raw: str | None) -> str:
    if raw in ADMIN_ROLES:
        return "admin"
    return "customer"


def _is_admin_role(raw: str | None) -> bool:
    return _normalize_role(raw) == "admin"


def _user_role_from_state(data: dict[str, Any]) -> str:
    return _normalize_role(data.get("user_role"))


def _session_get(user_id: int) -> dict[str, Any] | None:
    return CUSTOMER_SESSIONS.get(user_id)


def _checkout_link() -> str | None:
    username = settings.STORE_BOT_USERNAME
    if username:
        return f"https://t.me/{username}?start=checkout"
    return None


def _normalize_phone(raw: str | None) -> str | None:
    if not raw:
        return None
    digits = re.sub(r"\D", "", raw)
    if digits.startswith("0098"):
        digits = digits[4:]
    if digits.startswith("98"):
        digits = digits[2:]
    if digits.startswith("0") and len(digits) == 11:
        digits = digits[1:]
    if len(digits) == 10 and digits.startswith("9"):
        return "0" + digits
    if len(digits) == 11 and digits.startswith("0"):
        return digits
    return None


def _is_cancel_text(text: str | None) -> bool:
    if not text:
        return False
    return text.strip().lower() in {"لغو", "cancel", "انصراف", "/cancel"}


def _is_skip_text(text: str | None) -> bool:
    if not text:
        return False
    return text.strip().lower() in {"رد کردن", "skip"}


async def _send_account_contact_prompt(
    carrier: types.Message | types.CallbackQuery,
    state: FSMContext,
    text: str,
) -> None:
    if isinstance(carrier, types.CallbackQuery):
        await carrier.answer()
        chat_id = carrier.message.chat.id
        bot = carrier.bot
    else:
        chat_id = carrier.chat.id
        bot = carrier.bot
    await _send_clean_message(
        state,
        bot,
        chat_id,
        text,
        keyboards.account_contact_keyboard(),
        "account_prompt_message",
    )


async def _sync_session_to_state(state: FSMContext, user_id: int) -> None:
    session = _session_get(user_id)
    if not session:
        return
    role = _normalize_role(session.get("role"))
    tenant_id = session.get("tenant_id")
    customer_token = _session_customer_token(session)
    admin_token = _session_admin_token(session)
    updates: dict[str, Any] = {
        "user_role": role,
        "cust_session_token": session.get("session_token"),
    }
    if customer_token:
        updates["cust_token"] = customer_token
    if tenant_id:
        updates["cust_tenant_id"] = tenant_id
    if _is_admin_role(role):
        if admin_token:
            updates["token"] = admin_token
        if tenant_id:
            updates["tenant_id"] = tenant_id
    await state.update_data(**updates)


async def _finalize_checkout_summary(
    state: FSMContext,
    user_id: int,
    tenant_id: str,
    session_token: str,
    bot: types.Bot,
    chat_id: int,
) -> bool:
    client = BackendClient()
    try:
        await client.post(
            f"/visits/{session_token}/complete",
            {"status": "completed"},
            tenant_id=tenant_id,
        )
        likes = await client.get(f"/visits/{session_token}/likes", tenant_id=tenant_id)
    except Exception as exc:
        await bot.send_message(chat_id, f"خطا در تکمیل بازدید: {exc}")
        return False
    _session_update(user_id, session_token=None)
    selection = await _build_selection_from_likes(likes, tenant_id)
    if not selection:
        await bot.send_message(chat_id, "محصولی برای این بازدید ثبت نشد.")
        return False
    await state.update_data(cust_selection=selection)
    text = _selection_text(selection)
    markup = _selection_keyboard(selection)
    await bot.send_message(chat_id, text, reply_markup=markup)
    added = await _commit_selection_to_cart(state, user_id, selection, tenant_id)
    if added:
        await bot.send_message(chat_id, "محصولات به سبد خرید اضافه شدند.")
    return True


async def _get_tenant_profile(state: FSMContext) -> dict[str, Any] | None:
    data = await state.get_data()
    tenant_id = data.get("tenant_id")
    if not tenant_id:
        return None
    client = await _get_client(state)
    try:
        profile = await client.get(f"/tenants/{tenant_id}/profile", tenant_id=tenant_id)
    except Exception:
        return None
    return profile


async def _load_admin_categories(state: FSMContext) -> list[dict[str, Any]]:
    try:
        client = await _get_client(state)
    except RuntimeError:
        return []
    data = await state.get_data()
    tenant_id = data.get("tenant_id")
    if not tenant_id:
        return []
    try:
        categories = await client.get("/categories", tenant_id=tenant_id)
    except Exception:
        return []
    return categories or []


async def _get_public_channel_info(tenant_id: str | None) -> tuple[str | None, str | None]:
    if not tenant_id:
        return None, None
    client = BackendClient()
    try:
        info = await client.get(f"/public/tenants/{tenant_id}/channel")
    except Exception:
        return None, None
    if not isinstance(info, dict):
        return None, None
    return info.get("channel_id"), info.get("channel_invite_link")


def _resolve_send_target(target: types.Message | types.CallbackQuery) -> tuple[types.Message, types.Bot, int]:
    if isinstance(target, types.CallbackQuery):
        return target.message, target.bot, target.message.chat.id
    return target, target.bot, target.chat.id


async def _finalize_previous_visit_if_any(
    target: types.Message | types.CallbackQuery,
    state: FSMContext,
    telegram_user_id: int,
    tenant_id: str | None,
) -> None:
    if not tenant_id:
        return
    data = await state.get_data()
    session_token = data.get("cust_session_token")
    if not session_token:
        session = _session_get(telegram_user_id)
        if session:
            session_token = session.get("session_token")
            tenant_id = tenant_id or session.get("tenant_id")
    if not (session_token and tenant_id):
        return
    _, bot, chat_id = _resolve_send_target(target)
    await _finalize_checkout_summary(state, telegram_user_id, tenant_id, session_token, bot, chat_id)


async def _ensure_channel_member(
    target: types.Message | types.CallbackQuery,
    tenant_id: str | None,
    telegram_user_id: int,
) -> bool:
    if not tenant_id:
        return False
    channel_id, channel_link = await _get_public_channel_info(tenant_id)
    if not channel_id:
        return True
    send_target, bot, _ = _resolve_send_target(target)
    join_url = channel_link or channel_id or settings.STORE_CHANNEL_URL or "https://t.me"
    markup = types.InlineKeyboardMarkup(
        inline_keyboard=[
            [types.InlineKeyboardButton(text="🚪 ورود به کانال", url=join_url)]
        ]
    )
    try:
        member = await bot.get_chat_member(channel_id, telegram_user_id)
    except (TelegramBadRequest, TelegramForbidden):
        await send_target.answer(
            "برای ادامه باید عضو کانال فروشگاه باشید.",
            reply_markup=markup,
            disable_web_page_preview=True,
        )
        return False
    except Exception as exc:
        logger.exception("channel membership check failed: %s", exc)
        await send_target.answer(
            "امکان بررسی عضویت کانال وجود ندارد. لطفاً پس از بررسی دستی وارد ربات شوید.",
            disable_web_page_preview=True,
        )
        return False
    if member.status in {"left", "kicked"}:
        await send_target.answer(
            "برای ادامه باید عضو کانال فروشگاه باشید.",
            reply_markup=markup,
            disable_web_page_preview=True,
        )
        return False
    return True


async def _ensure_customer_session(state: FSMContext, telegram_user_id: int) -> bool:
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        return False
    if data.get("cust_token"):
        return True
    client = BackendClient(telegram_id=telegram_user_id)
    try:
        auth = await client.post("/auth/telegram-login", {}, tenant_id=tenant_id)
    except Exception:
        return False
    role_label = _normalize_role(auth.get("role"))
    updates = {
        "cust_token": auth["token"],
        "cust_tenant_id": auth["tenant_id"],
        "user_role": role_label,
    }
    if _is_admin_role(role_label):
        updates["token"] = auth["token"]
        updates["tenant_id"] = auth["tenant_id"]
    await state.update_data(**updates)
    _session_update(
        telegram_user_id,
        token=auth["token"],
        tenant_id=auth["tenant_id"],
        role=role_label,
        token_scope="customer",
    )
    return True


async def _get_customer_token(state: FSMContext, telegram_user_id: int) -> str | None:
    data = await state.get_data()
    token = data.get("cust_token")
    if token:
        return token
    if await _ensure_customer_session(state, telegram_user_id):
        data = await state.get_data()
        return data.get("cust_token")
    return None


def _parse_admin_ids(raw: str | None) -> list[int]:
    if not raw:
        return []
    values: list[int] = []
    for chunk in raw.split(","):
        chunk = chunk.strip()
        if not chunk:
            continue
        try:
            num = int(chunk)
        except ValueError:
            continue
        if num > 0:
            values.append(num)
    return values


def _normalize_manager_ids(raw: list[int | str] | None) -> list[int]:
    if not raw:
        return []
    normalized: list[int] = []
    for value in raw:
        try:
            num = int(value)
        except (TypeError, ValueError):
            continue
        if num > 0:
            normalized.append(num)
    return normalized


async def _get_admin_client() -> BackendClient:
    username = settings.DEFAULT_HYPER_ADMIN_USERNAME
    password = settings.DEFAULT_HYPER_ADMIN_PASSWORD
    if not (username and password):
        raise RuntimeError("hyper admin credentials are not configured")
    client = BackendClient()
    auth = await client.login(username, password)
    client.token = auth["token"]
    return client


async def _load_store_profile(tenant_id: str) -> dict[str, Any] | None:
    if not tenant_id:
        return None
    try:
        admin = await _get_admin_client()
        profile = await admin.get(f"/tenants/{tenant_id}/profile", tenant_id=tenant_id)
    except Exception as exc:
        logger.exception("failed to load tenant profile: %s", exc)
        return None
    return profile


async def _handle_channel_entry(
    target: types.Message | types.CallbackQuery,
    state: FSMContext,
    telegram_user_id: int,
) -> bool:
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        if isinstance(target, types.CallbackQuery):
            await target.answer("تنظیم STORE_TENANT_ID الزامی است.", show_alert=True)
        else:
            await target.answer("تنظیم STORE_TENANT_ID الزامی است.")
        return False
    await _finalize_previous_visit_if_any(target, state, telegram_user_id, tenant_id)
    allowed = await _ensure_channel_member(target, tenant_id, telegram_user_id)
    if not allowed:
        return False
    visit = await _start_customer_visit(state, telegram_user_id)
    if not visit:
        if isinstance(target, types.CallbackQuery):
            await target.answer("تنظیم STORE_TENANT_ID الزامی است.", show_alert=True)
        else:
            await target.answer("تنظیم STORE_TENANT_ID الزامی است.")
        return False
    logged_in = await _ensure_customer_session(state, telegram_user_id)
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    channel_id, channel_link = await _get_public_channel_info(tenant_id)
    link_to_show = channel_link or channel_id or settings.STORE_CHANNEL_URL or "کانال فروشگاه"
    if isinstance(target, types.CallbackQuery):
        await target.answer()
        send_target = target.message
    else:
        send_target = target
    entry_markup = types.InlineKeyboardMarkup(
        inline_keyboard=[
            [types.InlineKeyboardButton(text="🚪 ورود به کانال", url=link_to_show)],
        ]
    )
    await send_target.answer(
        "بازدید شما آغاز شد. وارد کانال شوید و پس از بازگشت، علاقه‌مندی‌ها شما به‌طور خودکار بروزرسانی می‌شوند.",
        reply_markup=entry_markup,
        disable_web_page_preview=True,
    )
    session_token = visit.get("session_token")
    _session_update(
        telegram_user_id,
        session_token=session_token,
        tenant_id=tenant_id,
        token=data.get("cust_token"),
        token_scope="customer",
    )
    await state.update_data(cust_session_token=session_token)
    return True


async def _show_account_menu(target: types.Message | types.CallbackQuery, state: FSMContext) -> None:
    data = await state.get_data()
    logged = bool(data.get("cust_token"))
    markup = keyboards.customer_account_menu(logged)
    text = "لطفاً یکی از گزینه‌های حساب کاربری را انتخاب کنید."
    if isinstance(target, types.CallbackQuery):
        await target.answer()
        await target.message.edit_text(text, reply_markup=markup)
        await state.update_data(account_prompt_message=target.message.message_id)
    else:
        await _send_clean_message(state, target.bot, target.chat.id, text, markup, "account_prompt_message")


async def _commit_selection_to_cart(
    state: FSMContext,
    telegram_user_id: int,
    selection: list[dict],
    tenant_id: str | None,
) -> bool:
    if not selection:
        return False
    data = await state.get_data()
    token = data.get("cust_token")
    if not token or not tenant_id:
        session = _session_get(telegram_user_id)
        if session:
            token = token or _session_customer_token(session)
            tenant_id = tenant_id or session.get("tenant_id")
    if not (token and tenant_id):
        return False
    client = BackendClient(token=token, telegram_id=telegram_user_id)
    try:
        for item in selection:
            await client.post(
                "/cart/items",
                {
                    "product_id": item["product_id"],
                    "qty": item["qty"],
                    "chosen_specs": item.get("chosen_specs") or {},
                },
                tenant_id=tenant_id,
            )
    except Exception as exc:
        logger.exception("auto add to cart failed: %s", exc)
        return False
    return True


async def _advance_category_step(state: FSMContext, selection: list[str]) -> None:
    data = await state.get_data()
    product_data = data.get("product_data", {})
    product_data["categories"] = selection
    idx = data.get("product_index", 0)
    await state.update_data(
        product_data=product_data,
        product_waiting=None,
        product_waiting_type=None,
        product_index=idx + 1,
    )
    await state.set_state(ProductWizardStates.flow)


async def _send_clean_message(
    state: FSMContext,
    bot: types.Bot,
    chat_id: int,
    text: str,
    reply_markup: types.InlineKeyboardMarkup | types.ReplyKeyboardMarkup | None = None,
    state_key: str = "last_bot_message_id",
) -> types.Message:
    data = await state.get_data()
    last_id = data.get(state_key)
    if last_id:
        try:
            await bot.delete_message(chat_id, last_id)
        except Exception:
            pass
    sent = await bot.send_message(chat_id, text, reply_markup=reply_markup)
    await state.update_data(**{state_key: sent.message_id})
    return sent


async def _show_loader(message: types.Message, text: str) -> types.Message:
    spinner = await message.answer(f"⌛ {text}")
    return spinner


def _format_cart_summary(order: dict[str, Any]) -> str:
    items = order.get("items") or []
    if not items:
        return "🛒 سبد خرید خالی است."
    lines = []
    for idx, item in enumerate(items, 1):
        price = item.get("final_price_snapshot") or 0
        qty = item.get("qty") or 0
        label = item.get("title") or f"#{item.get('product_id')}"
        lines.append(f"{idx}. {label} – تعداد {qty} – مبلغ {price:,.0f}")
    total = order.get("total") or 0
    status = order.get("status") or "cart"
    return "خلاصه سبد خرید:\n" + "\n".join(lines) + f"\n\nوضعیت: {status}\nجمع کل: {total:,.0f}"


def _format_product_caption(product: dict[str, Any]) -> str:
    title = product.get("title") or "محصول جدید"
    price = product.get("current_price") or product.get("base_price") or 0
    discount = product.get("discount_percent") or 0
    stock = product.get("stock")
    parts = [f"🌟 {title}", f"💰 قیمت: {price:,.0f}"]
    if discount:
        parts.append(f"🎯 تخفیف: {discount}%")
    if stock is not None:
        parts.append(f"📦 موجودی: {stock}")
    description = product.get("description")
    if description:
        parts.append("\n" + description.strip())
    parts.append("\nبرای سفارش از طریق ربات اقدام کنید.")
    return "\n".join(parts)


def _customer_menu_markup(data: dict[str, Any]) -> types.InlineKeyboardMarkup:
    logged = bool(data.get("cust_token"))
    return keyboards.customer_main_menu(logged)


async def _start_customer_visit(state: FSMContext, telegram_user_id: int) -> dict[str, Any] | None:
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        return None
    client = BackendClient()
    visit = await client.post("/visits/start", {"telegram_user_id": telegram_user_id}, tenant_id=tenant_id)
    await state.update_data(cust_session_token=visit["session_token"])
    return visit


async def _complete_customer_visit(state: FSMContext, user_id: int | None = None) -> tuple[list[dict], bool]:
    data = await state.get_data()
    session_token = data.get("cust_session_token")
    tenant_id = _customer_tenant_id(data)
    if user_id and not (session_token and tenant_id):
        session = _session_get(user_id)
        if session:
            session_token = session_token or session.get("session_token")
            tenant_id = tenant_id or session.get("tenant_id")
    if not (session_token and tenant_id):
        return [], False
    client = BackendClient()
    await client.post(f"/visits/{session_token}/complete", {"status": "completed"}, tenant_id=tenant_id)
    likes = await client.get(f"/visits/{session_token}/likes", tenant_id=tenant_id)
    await state.update_data(cust_session_token=None)
    if user_id:
        _session_update(user_id, session_token=None)
    return likes, True


def _format_likes_summary(likes: list[dict]) -> str:
    if not likes:
        return "در این بازدید محصولی نشانه‌گذاری نشد."
    lines = [
        f"{idx + 1}. محصول #{like['product_id']} – {like['liked_at']}"
        for idx, like in enumerate(likes)
    ]
    return "محصولات نشانه‌گذاری‌شده:\n" + "\n".join(lines)


def _format_order_history(orders: list[dict]) -> str:
    if not orders:
        return "هنوز سفارشی ثبت نکرده‌اید."
    lines = []
    for idx, order in enumerate(orders, 1):
        order_id = order.get("id", "")[:8]
        status = order.get("status")
        total = order.get("total", 0)
        lines.append(f"{idx}. سفارش #{order_id} – وضعیت: {status} – جمع: {total:,.0f}")
    return "سوابق سفارش:\n" + "\n".join(lines)


def _format_order_tracking(orders: list[dict]) -> str:
    active = [o for o in orders if o.get("status") not in {"paid", "canceled"}]
    if not active:
        return "سفارش فعالی برای پیگیری وجود ندارد."
    lines = []
    for order in active:
        order_id = order.get("id", "")[:8]
        status = order.get("status")
        items = order.get("items") or []
        lines.append(
            f"سفارش #{order_id} – وضعیت: {status} – اقلام: {len(items)}"
        )
    return "سفارش‌های در حال پیگیری:\n" + "\n".join(lines)


def _selection_text(selection: list[dict]) -> str:
    if not selection:
        return "هیچ محصولی در فهرست انتخابی نیست."
    lines = [
        f"{idx + 1}. {item['title']} × {item['qty']} (قیمت: {item['price']})"
        for idx, item in enumerate(selection)
    ]
    return "فهرست محصولات انتخابی:\n" + "\n".join(lines)


def _selection_keyboard(selection: list[dict]) -> InlineKeyboardMarkup:
    rows: list[list[InlineKeyboardButton]] = []
    for item in selection:
        pid = item["product_id"]
        rows.append(
            [
                InlineKeyboardButton(text="➖", callback_data=f"cust:sel:dec:{pid}"),
                InlineKeyboardButton(text=f"{item['title']} × {item['qty']}", callback_data="cust:sel:noop"),
                InlineKeyboardButton(text="➕", callback_data=f"cust:sel:inc:{pid}"),
            ]
        )
        rows.append([InlineKeyboardButton(text="🗑 حذف", callback_data=f"cust:sel:remove:{pid}")])
    if not rows:
        rows = [[InlineKeyboardButton(text="⬅️ منوی فروشگاه", callback_data="cust:menu")]]
    else:
        rows.append([InlineKeyboardButton(text="✅ افزودن به سبد خرید", callback_data="cust:sel:commit")])
        rows.append([InlineKeyboardButton(text="❌ حذف فهرست", callback_data="cust:sel:clear")])
        rows.append([InlineKeyboardButton(text="⬅️ منوی فروشگاه", callback_data="cust:menu")])
    return InlineKeyboardMarkup(inline_keyboard=rows)


def _favorites_text(favorites: list[dict]) -> str:
    if not favorites:
        return "لیست علاقه‌مندی‌ها خالی است."
    lines = [
        f"{idx + 1}. {item['title']} × {item['qty']} (قیمت: {item['price']})"
        for idx, item in enumerate(favorites)
    ]
    return "لیست علاقه‌مندی‌ها:\n" + "\n".join(lines)


def _favorites_keyboard(favorites: list[dict]) -> InlineKeyboardMarkup:
    rows: list[list[InlineKeyboardButton]] = []
    for item in favorites:
        pid = item["product_id"]
        rows.append(
            [
                InlineKeyboardButton(text="🗑 حذف", callback_data=f"fav:remove:{pid}"),
                InlineKeyboardButton(text="🛒 افزودن به سبد", callback_data=f"fav:cart:{pid}"),
            ]
        )
    if not rows:
        rows = [[InlineKeyboardButton(text="⬅️ منوی فروشگاه", callback_data="cust:menu")]]
    else:
        rows.append([InlineKeyboardButton(text="⬅️ منوی فروشگاه", callback_data="cust:menu")])
    return InlineKeyboardMarkup(inline_keyboard=rows)


async def _load_customer_favorites(
    tenant_id: str,
    token: str,
    telegram_user_id: int,
) -> list[dict]:
    client = BackendClient(token=token, telegram_id=telegram_user_id)
    likes = await client.get("/visits/me/likes", tenant_id=tenant_id)
    return await _build_selection_from_likes(likes, tenant_id)


async def _render_customer_favorites(
    callback: types.CallbackQuery,
    state: FSMContext,
    favorites: list[dict],
) -> None:
    await state.update_data(cust_favorites=favorites)
    text = _favorites_text(favorites)
    markup = _favorites_keyboard(favorites)
    await callback.message.edit_text(text, reply_markup=markup)


async def _enrich_cart_items(
    items: list[dict],
    tenant_id: str,
    token: str,
    telegram_user_id: int | None = None,
) -> list[dict]:
    if not items:
        return []
    client = BackendClient(token=token, telegram_id=telegram_user_id)
    semaphore = asyncio.Semaphore(6)

    async def _fetch(item: dict) -> dict:
        enriched = item.copy()
        pid = enriched.get("product_id")
        if not pid:
            return enriched
        async with semaphore:
            try:
                product = await client.get(f"/products/{pid}", tenant_id=tenant_id)
            except Exception:
                product = None
        title = product.get("title") if isinstance(product, dict) else None
        if title:
            enriched["title"] = title
        return enriched

    tasks = [_fetch(item) for item in items]
    return await asyncio.gather(*tasks, return_exceptions=False)


def _cart_items_keyboard(items: list[dict]) -> InlineKeyboardMarkup:
    rows: list[list[InlineKeyboardButton]] = []
    for item in items:
        item_id = item.get("id")
        label = item.get("title") or str(item.get("product_id"))
        if not item_id:
            continue
        rows.append(
            [
                InlineKeyboardButton(
                    text=f"🗑 حذف {label[:24]}",
                    callback_data=f"cart:item:remove:{item_id}",
                ),
                InlineKeyboardButton(
                    text=f"✅ تکمیل {label[:24]}",
                    callback_data=f"cart:item:complete:{item_id}",
                ),
            ]
        )
    rows.append([InlineKeyboardButton(text="✅ نهایی‌سازی سفارش", callback_data="cust:checkout")])
    rows.append([InlineKeyboardButton(text="🗑 حذف سبد", callback_data="cust:cart_cancel")])
    rows.append([InlineKeyboardButton(text="⬅️ بازگشت به منو", callback_data="cust:menu")])
    return InlineKeyboardMarkup(inline_keyboard=rows)


async def _render_cart_summary(
    state: FSMContext,
    bot: types.Bot,
    chat_id: int,
    order: dict[str, Any],
    tenant_id: str,
    token: str,
    telegram_user_id: int,
) -> None:
    items = order.get("items") or []
    enriched = await _enrich_cart_items(items, tenant_id, token, telegram_user_id)
    order["items"] = enriched
    text = _format_cart_summary(order)
    markup = _cart_items_keyboard(enriched)
    await _send_clean_message(
        state,
        bot,
        chat_id,
        text,
        markup,
        state_key="last_cart_message_id",
    )


async def _build_selection_from_likes(likes: list[dict], tenant_id: str | None) -> list[dict]:
    if not tenant_id or not likes:
        return []
    client = BackendClient()
    semaphore = asyncio.Semaphore(10)

    async def _fetch_product(like: dict) -> dict | None:
        pid = like["product_id"]
        async with semaphore:
            try:
                product = await client.get(f"/products/{pid}", tenant_id=tenant_id)
            except Exception:
                return None
        if not product:
            return None
        return {
            "product_id": pid,
            "title": product.get("title"),
            "qty": 1,
            "price": product.get("current_price") or product.get("base_price") or 0,
        }

    tasks = [_fetch_product(like) for like in likes]
    results = await asyncio.gather(*tasks, return_exceptions=False)
    return [item for item in results if item]


async def _reset_product_wizard(state: FSMContext) -> None:
    await state.update_data(
        product_index=0,
        product_data={},
        product_waiting=None,
        product_waiting_type=None,
        product_categories=[],
        available_categories=[],
    )


async def _current_product_step(state: FSMContext) -> dict[str, Any] | None:
    data = await state.get_data()
    idx = data.get("product_index", 0)
    if idx >= len(PRODUCT_FLOW):
        return None
    return PRODUCT_FLOW[idx]


async def _show_product_prompt(target: types.Message | types.CallbackQuery, state: FSMContext) -> None:
    step = await _current_product_step(state)
    data = await state.get_data()
    product_data: dict[str, Any] = data.get("product_data", {})
    idx = data.get("product_index", 0)

    if isinstance(target, types.CallbackQuery):
        msg = target.message
    else:
        msg = target

    if step is None:
        lines: list[str] = []
        cat_lookup = {c["id"]: c["name"] for c in data.get("available_categories", []) if isinstance(c, dict)}
        for key, value in product_data.items():
            label = PRODUCT_MAP.get(key, {}).get("label", key)
            if key == "categories":
                names = [cat_lookup.get(cid, cid) for cid in value or []]
                value = ", ".join(names) if names else "بدون دسته"
            lines.append(f"{label}: {value}")
        summary = "پیش‌نمایش محصول:\n" + "\n".join(lines)
        await msg.answer(summary, reply_markup=keyboards.wizard_back_cancel("prod"))
        await msg.answer("برای تایید، دکمه زیر را بزنید.", reply_markup=types.InlineKeyboardMarkup(
            inline_keyboard=[
                [types.InlineKeyboardButton(text="✅ تایید محصول", callback_data="prod:confirm")],
                [
                    types.InlineKeyboardButton(text="⬅️ بازگشت", callback_data="prod:back"),
                    types.InlineKeyboardButton(text="❌ لغو", callback_data="prod:cancel"),
                ],
            ]
        ))
        return

    text = f"مرحله {idx + 1}/{len(PRODUCT_FLOW)} – {step['label']}"
    current_value = product_data.get(step["key"])
    if current_value:
        text += f"\nمقدار فعلی: {current_value}"
    if step.get("type") == "categories":
        categories = await _load_admin_categories(state)
        if not categories:
            await msg.answer("هیچ دسته‌ای تعریف نشده است؛ این مرحله رد شد.", reply_markup=keyboards.wizard_back_cancel("prod"))
            await state.update_data(product_index=idx + 1)
            await state.set_state(ProductWizardStates.flow)
            await _show_product_prompt(target, state)
            return
        selected = data.get("product_categories", [])
        await state.update_data(
            product_waiting="categories",
            product_waiting_type="categories",
            available_categories=categories,
        )
        await state.set_state(ProductWizardStates.awaiting_value)
        await msg.answer(
            "دسته‌بندی‌های مرتبط را انتخاب کنید و در پایان «✅ تایید انتخاب» یا «⛔ بدون دسته» را بزنید.",
            reply_markup=keyboards.product_categories_wizard(categories, selected),
        )
        return
    type_hints = {
        "text": "لطفاً مقدار را تایپ کنید.",
        "number": "یک مقدار عددی ارسال کنید.",
        "photo": "لطفاً تصویر محصول را به صورت Photo بفرستید.",
    }
    text += f"\n\n{type_hints.get(step.get('type'), 'مقدار را ارسال کنید.')}"
    await msg.answer(text, reply_markup=keyboards.wizard_back_cancel("prod"))
    await state.set_state(ProductWizardStates.awaiting_value)
    await state.update_data(
        product_waiting=step["key"],
        product_waiting_type=step.get("type", "text"),
    )


async def _send_product_list(callback: types.CallbackQuery, state: FSMContext, active: bool) -> None:
    client = await _get_client(state)
    tenant_id = _tenant_id_from_state(await state.get_data())
    prods = await client.get("/products", tenant_id=tenant_id, params={"active": active})
    if not prods:
        text = "محصولی یافت نشد."
    else:
        text = "\n".join(f"{p['title']} – {p['sku']}" for p in prods[:20])
    await callback.answer()
    await callback.message.answer(text)


# ---------------------------------------------------------------------------
# Entry points
# ---------------------------------------------------------------------------


@router.message(CommandStart())
async def start(message: types.Message, state: FSMContext):
    text = message.text or ""
    parts = text.split(maxsplit=1)
    payload = parts[1] if len(parts) > 1 else None
    await state.clear()
    await _sync_session_to_state(state, message.from_user.id)
    data = await state.get_data()
    role = _user_role_from_state(data)
    show_admin_entry = not _is_admin_role(role)
    await message.answer(
        "به ربات فروشگاهی خوش آمدید. برای مدیریت مارکت یا بازدید از فروشگاه، از دکمه‌های زیر استفاده کنید.",
        reply_markup=keyboards.landing_menu(show_admin_entry=show_admin_entry),
    )
    if _is_admin_role(role):
        await state.set_state(StoreLoginStates.admin_menu)
        await message.answer("شما در نقش مدیر فروشگاه هستید.", reply_markup=keyboards.admin_main_menu())
    else:
        await message.answer("منوی فروشگاه:", reply_markup=_customer_menu_markup(data))
    if payload == "checkout":
        session = _session_get(message.from_user.id)
        if not session or not session.get("session_token") or not session.get("tenant_id"):
            await message.answer("بازدید فعالی برای تکمیل خرید یافت نشد.")
            return
        await _finalize_checkout_summary(
            state,
            message.from_user.id,
            session["tenant_id"],
            session["session_token"],
            message.bot,
            message.chat.id,
        )


@router.message(F.text.in_(["مدیریت مارکت", "🔐 مدیریت مارکت"]))
async def admin_login_start(message: types.Message, state: FSMContext):
    await state.set_state(StoreLoginStates.username)
    await message.answer("نام کاربری مدیر مارکت را وارد کنید:")


@router.message(StoreLoginStates.username)
async def admin_login_username(message: types.Message, state: FSMContext):
    await state.update_data(username=message.text.strip())
    await state.set_state(StoreLoginStates.password)
    await message.answer("رمز عبور را وارد کنید:")


@router.message(StoreLoginStates.password)
async def admin_login_password(message: types.Message, state: FSMContext):
    data = await state.get_data()
    client = BackendClient()
    waiting = await _show_loader(message, "در حال بررسی اطلاعات ورود...")
    try:
        auth = await client.login(data["username"], message.text.strip())
    except Exception as exc:
        await waiting.edit_text(f"ورود ناموفق بود: {exc}")
        await state.set_state(StoreLoginStates.username)
        logger.exception("store admin login failed")
        return
    role_label = _normalize_role(auth.get("role"))
    await state.update_data(
        token=auth["token"],
        tenant_id=auth.get("tenant_id"),
        user_role=role_label,
        cust_token=auth["token"],
        cust_tenant_id=auth.get("tenant_id"),
    )
    _session_update(
        message.from_user.id,
        token=auth["token"],
        tenant_id=auth.get("tenant_id"),
        role=role_label,
        token_scope="admin",
    )
    await state.set_state(StoreLoginStates.admin_menu)
    await waiting.edit_text("ورود موفق بود.")
    await message.answer("منوی مدیریت:", reply_markup=keyboards.admin_main_menu())


@router.message(F.text == "📢 مشاهده محصولات")
async def customer_view_products(message: types.Message, state: FSMContext):
    data = await state.get_data()
    if _is_admin_role(data.get("user_role")):
        await message.answer("شما در نقش مدیر هستید. برای مدیریت از منوی قبلی استفاده کنید.", reply_markup=keyboards.admin_main_menu())
        return
    success = await _handle_channel_entry(message, state, message.from_user.id)
    if not success:
        await message.answer("برای مشاهده محصولات، لطفاً ابتدا مشتری پیش‌فرض فروشگاه را تنظیم کنید.")


@router.message(F.text == "👤 حساب کاربری")
async def customer_account_entry(message: types.Message, state: FSMContext):
    data = await state.get_data()
    if _is_admin_role(data.get("user_role")):
        await message.answer("شما پیش از این به حساب مدیر وارد شده‌اید.", reply_markup=keyboards.admin_main_menu())
        return
    await state.set_state(CustomerAccountStates.menu)
    await _show_account_menu(message, state)


@router.message(StoreLoginStates.admin_menu, F.text == "📦 محصولات")
async def admin_products_menu(message: types.Message, state: FSMContext):
    await state.set_state(AdminFlowStates.section)
    await message.answer("لطفاً یک عملیات را انتخاب کنید:", reply_markup=keyboards.admin_products_menu())


@router.message(StoreLoginStates.admin_menu, F.text == "📁 دسته‌بندی‌ها")
async def admin_categories_menu(message: types.Message, state: FSMContext):
    await state.set_state(CategoryWizardStates.managing)
    await message.answer("مدیریت دسته‌بندی:", reply_markup=keyboards.admin_categories_menu())


@router.message(StoreLoginStates.admin_menu, F.text == "🧾 سفارش‌ها")
async def admin_orders_menu(message: types.Message, state: FSMContext):
    await state.set_state(OrderStates.viewing)
    await message.answer("مدیریت سفارش‌ها:", reply_markup=keyboards.admin_orders_menu())


@router.message(StoreLoginStates.admin_menu, F.text == "📣 کانال فروشگاه")
async def admin_channel_menu(message: types.Message, state: FSMContext):
    await state.set_state(ChannelStates.awaiting_channel_id)
    await message.answer("مدیریت کانال:", reply_markup=keyboards.admin_channel_menu())


@router.message(StoreLoginStates.admin_menu, F.text == "↩️ خروج")
async def admin_logout(message: types.Message, state: FSMContext):
    await state.clear()
    await state.update_data(user_role="customer")
    _session_update(
        message.from_user.id,
        token=None,
        tenant_id=None,
        role=None,
        session_token=None,
        token_scope="all",
    )
    data = await state.get_data()
    await message.answer("از حساب خارج شدید.", reply_markup=keyboards.landing_menu(show_admin_entry=True))
    await message.answer("منوی فروشگاه:", reply_markup=_customer_menu_markup(data))


# ---------------------------------------------------------------------------
# Product wizard callbacks
# ---------------------------------------------------------------------------


@router.callback_query(F.data == "admin:product:new")
async def product_wizard_start(callback: types.CallbackQuery, state: FSMContext):
    await callback.answer()
    await state.set_state(ProductWizardStates.flow)
    await _reset_product_wizard(state)
    await callback.message.answer("ویزارد ثبت محصول آغاز شد.")
    await _show_product_prompt(callback, state)


@router.callback_query(F.data.startswith("prod:back"))
async def product_wizard_back(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    idx = max(data.get("product_index", 0) - 1, 0)
    await state.update_data(product_index=idx)
    await callback.answer()
    await _show_product_prompt(callback, state)


@router.callback_query(F.data == "prod:cancel")
async def product_wizard_cancel(callback: types.CallbackQuery, state: FSMContext):
    await state.set_state(StoreLoginStates.admin_menu)
    await callback.answer("عملیات لغو شد.")
    await callback.message.answer("به منوی اصلی برگشتید.", reply_markup=keyboards.admin_main_menu())


@router.message(ProductWizardStates.awaiting_value, Command("cancel"))
async def product_wizard_cancel_command(message: types.Message, state: FSMContext):
    await state.set_state(StoreLoginStates.admin_menu)
    await message.answer("عملیات لغو شد.", reply_markup=keyboards.admin_main_menu())


@router.message(ProductWizardStates.awaiting_value, F.photo)
async def product_wizard_photo(message: types.Message, state: FSMContext):
    data = await state.get_data()
    waiting_key = data.get("product_waiting")
    waiting_type = data.get("product_waiting_type")
    if waiting_type != "photo":
        await message.answer("در این مرحله باید متن ارسال کنید.")
        return
    file_id = message.photo[-1].file_id
    product_data = data.get("product_data", {})
    product_data[waiting_key] = file_id
    await state.update_data(product_data=product_data, product_waiting=None, product_waiting_type=None)
    await state.set_state(ProductWizardStates.flow)
    idx = data.get("product_index", 0)
    await state.update_data(product_index=idx + 1)
    await _show_product_prompt(message, state)


@router.message(ProductWizardStates.awaiting_value)
async def product_wizard_value(message: types.Message, state: FSMContext):
    data = await state.get_data()
    waiting_key = data.get("product_waiting")
    waiting_type = data.get("product_waiting_type", "text")
    if not waiting_key:
        await message.answer("مرحله‌ای در انتظار مقدار نیست.")
        return
    value = message.text.strip()
    if waiting_type == "categories":
        text = (message.text or "").strip().lower()
        if "بدون" in text:
            await _advance_category_step(state, [])
            await message.answer("مرحله دسته‌بندی با گزینه «بدون دسته» رد شد.")
            await _show_product_prompt(message, state)
        elif "تایید" in text or "✅" in text:
            data = await state.get_data()
            selection: list[str] = data.get("product_categories", [])
            await _advance_category_step(state, selection)
            await message.answer("دسته‌بندی‌ها ثبت شد.")
            await _show_product_prompt(message, state)
        else:
            await message.answer("برای انتخاب دسته‌بندی از دکمه‌های موجود استفاده کنید.")
        return
    if waiting_type == "number":
        try:
            value = float(value) if waiting_key == "base_price" else int(value)
        except ValueError:
            await message.answer("ورودی باید عددی باشد.")
            return
    product_data = data.get("product_data", {})
    product_data[waiting_key] = value
    await state.update_data(product_data=product_data, product_waiting=None, product_waiting_type=None)
    idx = data.get("product_index", 0)
    await state.update_data(product_index=idx + 1)
    await state.set_state(ProductWizardStates.flow)
    await _show_product_prompt(message, state)


@router.callback_query(F.data.startswith("prod:cat:toggle"))
async def product_wizard_toggle_category(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    if data.get("product_waiting_type") != "categories":
        await callback.answer()
        return
    cat_id = callback.data.split(":", 3)[3]
    selected: list[str] = data.get("product_categories", [])
    if cat_id in selected:
        selected.remove(cat_id)
    else:
        selected.append(cat_id)
    categories = data.get("available_categories", [])
    await state.update_data(product_categories=selected)
    await callback.answer("به‌روزرسانی شد.")
    await callback.message.edit_reply_markup(
        reply_markup=keyboards.product_categories_wizard(categories, selected)
    )


@router.callback_query(F.data == "prod:cat:done")
async def product_wizard_categories_done(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    if data.get("product_waiting_type") != "categories":
        await callback.answer()
        return
    selection: list[str] = data.get("product_categories", [])
    await _advance_category_step(state, selection)
    await callback.answer("دسته‌ها ثبت شد.")
    await _show_product_prompt(callback, state)


@router.callback_query(F.data == "prod:cat:skip")
async def product_wizard_categories_skip(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    if data.get("product_waiting_type") != "categories":
        await callback.answer()
        return
    await _advance_category_step(state, [])
    await callback.answer("مرحله دسته‌بندی رد شد.")
    await _show_product_prompt(callback, state)


@router.callback_query(F.data == "prod:confirm")
async def product_wizard_confirm(callback: types.CallbackQuery, state: FSMContext):
    await callback.answer()
    data = await state.get_data()
    product_data = data.get("product_data", {})
    tenant_id = _tenant_id_from_state(data)
    client = await _get_client(state)
    payload = {
        "sku": product_data["title"].replace(" ", "-")[:16],
        "title": product_data["title"],
        "name": product_data["title"],
        "description": product_data.get("description"),
        "base_price": product_data.get("base_price"),
        "discount_percent": product_data.get("discount_percent", 0),
        "stock": product_data.get("stock", 0),
        "image_file_id": product_data.get("image_file_id"),
        "categories": product_data.get("categories", []),
        "specs": [],
    }
    waiting = await _show_loader(callback.message, "در حال ثبت محصول...")
    try:
        created = await client.post("/products", payload, tenant_id=tenant_id)
    except httpx.HTTPStatusError as exc:
        detail = ""
        if exc.response is not None:
            try:
                detail = exc.response.text
            except Exception:
                detail = str(exc.response)
        await waiting.edit_text(f"ثبت محصول ناموفق بود: {exc}. {detail}")
        logger.exception("product create failed")
        return
    except Exception as exc:
        await waiting.edit_text(f"ثبت محصول ناموفق بود: {exc}")
        logger.exception("product create failed")
        return
    await waiting.edit_text(f"محصول با موفقیت ثبت شد. SKU: {created['sku']}")
    profile = await _get_tenant_profile(state)
    channel_id = (profile or {}).get("channel_id")
    if not channel_id:
        channel_id, _ = await _get_public_channel_info(tenant_id)
    if not channel_id:
        channel_id = settings.STORE_CHANNEL_URL
    if channel_id:
        try:
            caption = _format_product_caption(created)
            post_markup = keyboards.product_channel_keyboard(created["id"], _checkout_link())
            if created.get("image_file_id"):
                await callback.bot.send_photo(
                    channel_id,
                    created["image_file_id"],
                    caption=caption,
                    reply_markup=post_markup,
                )
            else:
                await callback.bot.send_message(channel_id, caption, reply_markup=post_markup)
        except Exception as exc:
            logger.exception("channel publish failed")
            await callback.message.answer("محصول ثبت شد، اما ارسال به کانال با خطا مواجه شد. تنظیمات کانال را بررسی کنید.")
    else:
        await callback.message.answer("محصول ثبت شد، اما کانال برای این فروشگاه تنظیم نشده است.")
    await state.set_state(StoreLoginStates.admin_menu)


# ---------------------------------------------------------------------------
# Category management
# ---------------------------------------------------------------------------


@router.callback_query(F.data == "admin:cat:list")
async def admin_list_categories(callback: types.CallbackQuery, state: FSMContext):
    client = await _get_client(state)
    tenant_id = _tenant_id_from_state(await state.get_data())
    cats = await client.get("/categories", tenant_id=tenant_id)
    text = "\n".join(f"- {c['name']}" for c in cats) or "هیچ دسته‌ای ثبت نشده است."
    await callback.answer()
    await callback.message.answer(text)


@router.callback_query(F.data == "admin:cat:new")
async def admin_new_category(callback: types.CallbackQuery, state: FSMContext):
    await callback.answer()
    await state.set_state(CategoryWizardStates.awaiting_name)
    await callback.message.answer("نام دسته جدید را ارسال کنید. برای لغو، /cancel را بزنید.")


@router.message(CategoryWizardStates.awaiting_name)
async def admin_save_category(message: types.Message, state: FSMContext):
    name = message.text.strip()
    if not name:
        await message.answer("نام نمی‌تواند خالی باشد.")
        return
    client = await _get_client(state)
    tenant_id = _tenant_id_from_state(await state.get_data())
    try:
        created = await client.post("/categories", {"name": name}, tenant_id=tenant_id)
    except Exception as exc:
        await message.answer(f"خطا در ایجاد دسته: {exc}")
        logger.exception("category create failed")
        return
    await state.set_state(CategoryWizardStates.managing)
    await message.answer(f"دسته {created['name']} اضافه شد.", reply_markup=keyboards.admin_categories_menu())


@router.message(CategoryWizardStates.awaiting_name, Command("cancel"))
async def admin_cancel_category(message: types.Message, state: FSMContext):
    await state.set_state(CategoryWizardStates.managing)
    await message.answer("عملیات لغو شد.", reply_markup=keyboards.admin_categories_menu())


# ---------------------------------------------------------------------------
# Orders management
# ---------------------------------------------------------------------------


@router.callback_query(F.data.startswith("admin:orders:"))
async def admin_orders(callback: types.CallbackQuery, state: FSMContext):
    status = callback.data.split(":")[-1]
    client = await _get_client(state)
    tenant_id = _tenant_id_from_state(await state.get_data())
    orders = await client.get("/orders", tenant_id=tenant_id, params={"status": status})
    if not orders:
        text = "سفارشی یافت نشد."
    else:
        text = "\n".join(f"{o['id']} – {o['total']}" for o in orders)
    await callback.answer()
    await callback.message.answer(text)


@router.callback_query(F.data == "admin:product:list:active")
async def admin_products_list_active(callback: types.CallbackQuery, state: FSMContext):
    await _send_product_list(callback, state, True)


@router.callback_query(F.data == "admin:product:list:inactive")
async def admin_products_list_inactive(callback: types.CallbackQuery, state: FSMContext):
    await _send_product_list(callback, state, False)


@router.callback_query(F.data == "admin:back:main")
async def admin_back_main(callback: types.CallbackQuery, state: FSMContext):
    await state.set_state(StoreLoginStates.admin_menu)
    await callback.answer()
    await callback.message.answer("منوی اصلی:", reply_markup=keyboards.admin_main_menu())


@router.callback_query(F.data == "admin:cancel")
async def admin_cancel(callback: types.CallbackQuery, state: FSMContext):
    await state.set_state(StoreLoginStates.admin_menu)
    await callback.answer("عملیات لغو شد.", show_alert=True)
    await callback.message.answer("به منوی اصلی بازگشتید.", reply_markup=keyboards.admin_main_menu())


# ---------------------------------------------------------------------------
# Channel management
# ---------------------------------------------------------------------------


@router.callback_query(F.data == "admin:channel:set")
async def channel_set(callback: types.CallbackQuery, state: FSMContext):
    await callback.answer()
    await state.set_state(ChannelStates.awaiting_channel_id)
    await callback.message.answer("شناسه کانال (مثال: @mychannel) را وارد کنید. /cancel برای لغو.")


@router.callback_query(F.data == "admin:channel:test")
async def channel_test(callback: types.CallbackQuery, state: FSMContext):
    profile = await _get_tenant_profile(state)
    channel_id = profile.get("channel_id") if profile else None
    if not channel_id:
        await callback.answer("ابتدا channel_id را تنظیم کنید.", show_alert=True)
        return
    try:
        await callback.bot.send_message(channel_id, "پیام تست از ربات فروشگاه ✅")
    except Exception as exc:
        await callback.answer(f"ارسال ناموفق بود: {exc}", show_alert=True)
        return
    await callback.answer("پیام تست به کانال ارسال شد.", show_alert=True)


@router.message(ChannelStates.awaiting_channel_id, Command("cancel"))
async def channel_cancel(message: types.Message, state: FSMContext):
    await state.set_state(StoreLoginStates.admin_menu)
    await message.answer("عملیات تنظیم کانال لغو شد.", reply_markup=keyboards.admin_main_menu())


@router.message(ChannelStates.awaiting_channel_id)
async def channel_receive_id(message: types.Message, state: FSMContext):
    channel_id = message.text.strip()
    await state.update_data(channel_pending_id=channel_id)
    await state.set_state(ChannelStates.awaiting_invite)
    await message.answer("لینک دعوت کانال را وارد کنید.")


@router.message(ChannelStates.awaiting_invite)
async def channel_receive_invite(message: types.Message, state: FSMContext):
    data = await state.get_data()
    client = await _get_client(state)
    tenant_id = _tenant_id_from_state(data)
    channel_id = data.get("channel_pending_id")
    invite_link = message.text.strip()
    if not channel_id:
        await message.answer("ابتدا شناسه کانال را ارسال کنید.", reply_markup=keyboards.admin_channel_menu())
        return
    if not invite_link.startswith("http"):
        await message.answer("لطفاً لینک دعوت معتبر (با http/https) ارسال کنید.")
        return
    payload = {
        "channel_id": channel_id,
        "channel_invite_link": invite_link,
    }
    try:
        await client.put(f"/tenants/{tenant_id}/profile", payload, tenant_id=tenant_id)
    except Exception as exc:
        detail = getattr(getattr(exc, "response", None), "text", None)
        extra = f"\nجزئیات: {detail}" if detail else ""
        await message.answer(f"به‌روزرسانی کانال ناموفق بود: {exc}{extra}")
        logger.exception("channel update failed")
        return
    await state.set_state(StoreLoginStates.admin_menu)
    await message.answer("اطلاعات کانال ذخیره شد.", reply_markup=keyboards.admin_main_menu())


# ---------------------------------------------------------------------------
# Customer account flow
# ---------------------------------------------------------------------------


@router.callback_query(F.data.startswith("acct:"))
async def customer_account_callbacks(callback: types.CallbackQuery, state: FSMContext):
    action = callback.data.split(":", 1)[1]
    data = await state.get_data()
    if action == "back":
        await state.set_state(CustomerAccountStates.menu)
        await _show_account_menu(callback, state)
        return
    if action == "login":
        await state.set_state(CustomerAccountStates.login_contact)
        await _send_account_contact_prompt(
            callback,
            state,
            "لطفاً شماره موبایل خود را از طریق دکمه زیر ارسال کنید.",
        )
        return
    if action == "register":
        await state.update_data(account_register={})
        await state.set_state(CustomerAccountStates.register_contact)
        await _send_account_contact_prompt(
            callback,
            state,
            "برای ساخت حساب، ابتدا شماره موبایل خود را ارسال کنید.",
        )
        return
    if action == "cart":
        token = data.get("cust_token")
        tenant_id = _customer_tenant_id(data)
        if not (token and tenant_id):
            await callback.answer("برای مشاهده سبد ابتدا وارد شوید.", show_alert=True)
            return
        client = BackendClient(token=token, telegram_id=callback.from_user.id)
        order = await client.post("/cart/start", {}, tenant_id=tenant_id)
        text = _format_cart_summary(order)
        await callback.answer()
        await callback.message.answer(text, reply_markup=keyboards.customer_cart_kb())
        return
    if action in {"history", "track"}:
        token = data.get("cust_token")
        tenant_id = _customer_tenant_id(data)
        if not (token and tenant_id):
            await callback.answer("ابتدا وارد حساب شوید.", show_alert=True)
            return
        client = BackendClient(token=token, telegram_id=callback.from_user.id)
        try:
            orders = await client.get("/orders/my", tenant_id=tenant_id)
        except Exception as exc:
            await callback.answer(f"خطا در دریافت سفارش‌ها: {exc}", show_alert=True)
            return
        text = (
            _format_order_tracking(orders)
            if action == "track"
            else _format_order_history(orders)
        )
        await callback.answer()
        await _send_clean_message(
            state,
            callback.bot,
            callback.message.chat.id,
            text,
            keyboards.customer_account_menu(True),
            "account_prompt_message",
        )
        return


@router.callback_query(F.data.in_(["acctlogin:back", "acctlogin:cancel"]))
async def account_login_nav(callback: types.CallbackQuery, state: FSMContext):
    await state.set_state(CustomerAccountStates.menu)
    try:
        await callback.message.answer("عملیات لغو شد.", reply_markup=types.ReplyKeyboardRemove())
    except Exception:
        pass
    await _show_account_menu(callback, state)


@router.callback_query(F.data.in_(["acctreg:back", "acctreg:cancel"]))
async def account_register_nav(callback: types.CallbackQuery, state: FSMContext):
    await state.set_state(CustomerAccountStates.menu)
    try:
        await callback.message.answer("عملیات لغو شد.", reply_markup=types.ReplyKeyboardRemove())
    except Exception:
        pass
    await _show_account_menu(callback, state)


@router.message(CustomerAccountStates.login_contact, F.contact)
async def account_login_contact_shared(message: types.Message, state: FSMContext):
    phone = _normalize_phone(message.contact.phone_number)
    if not phone:
        await _send_account_contact_prompt(
            message,
            state,
            "شماره نامعتبر است. لطفاً از دکمه «ارسال شماره من» استفاده کنید.",
        )
        return
    await state.update_data(account_login_phone=phone)
    await state.set_state(CustomerAccountStates.login_password)
    await _send_clean_message(
        state,
        message.bot,
        message.chat.id,
        "رمز عبور را وارد کنید:",
        types.ReplyKeyboardRemove(),
        "account_prompt_message",
    )


@router.message(CustomerAccountStates.login_contact)
async def account_login_contact_text(message: types.Message, state: FSMContext):
    if _is_cancel_text(message.text):
        await state.set_state(CustomerAccountStates.menu)
        await _show_account_menu(message, state)
        return
    phone = _normalize_phone(message.text)
    if not phone:
        await _send_account_contact_prompt(
            message,
            state,
            "شماره وارد شده معتبر نیست. لطفاً با فرمت صحیح یا با دکمه ارسال شماره وارد شوید.",
        )
        return
    await state.update_data(account_login_phone=phone)
    await state.set_state(CustomerAccountStates.login_password)
    await _send_clean_message(
        state,
        message.bot,
        message.chat.id,
        "رمز عبور را وارد کنید:",
        types.ReplyKeyboardRemove(),
        "account_prompt_message",
    )


@router.message(CustomerAccountStates.login_password)
async def account_login_password(message: types.Message, state: FSMContext):
    if _is_cancel_text(message.text):
        await state.set_state(CustomerAccountStates.menu)
        await _show_account_menu(message, state)
        return
    data = await state.get_data()
    phone = data.get("account_login_phone")
    if not phone:
        await state.set_state(CustomerAccountStates.login_contact)
        await _send_account_contact_prompt(
            message,
            state,
            "ابتدا شماره موبایل را ارسال کنید.",
        )
        return
    client = BackendClient()
    waiting = await _show_loader(message, "در حال ورود به حساب...")
    try:
        auth = await client.login(phone, message.text.strip())
    except Exception as exc:
        await waiting.edit_text(f"ورود ناموفق بود: {exc}")
        return
    await waiting.edit_text("ورود موفق بود.")
    role_label = _normalize_role(auth.get("role"))
    await state.update_data(
        cust_token=auth["token"],
        cust_tenant_id=auth.get("tenant_id"),
        account_login_phone=None,
        user_role=role_label,
    )
    _session_update(
        message.from_user.id,
        token=auth["token"],
        tenant_id=auth.get("tenant_id"),
        role=role_label,
        token_scope="customer",
    )
    await state.set_state(CustomerAccountStates.menu)
    await _show_account_menu(message, state)


async def _update_registration(state: FSMContext, **kwargs: Any) -> None:
    data = await state.get_data()
    reg: dict[str, Any] = data.get("account_register") or {}
    reg.update(kwargs)
    await state.update_data(account_register=reg)


@router.message(CustomerAccountStates.register_contact, F.contact)
async def account_register_contact_shared(message: types.Message, state: FSMContext):
    phone = _normalize_phone(message.contact.phone_number)
    if not phone:
        await _send_account_contact_prompt(
            message,
            state,
            "شماره نامعتبر است. لطفاً از دکمه «ارسال شماره من» استفاده کنید.",
        )
        return
    await _update_registration(state, phone=phone)
    await state.set_state(CustomerAccountStates.register_first)
    await _send_clean_message(
        state,
        message.bot,
        message.chat.id,
        "نام کوچک خود را وارد کنید:",
        types.ReplyKeyboardRemove(),
        "account_prompt_message",
    )


@router.message(CustomerAccountStates.register_contact)
async def account_register_contact_text(message: types.Message, state: FSMContext):
    if _is_cancel_text(message.text):
        await state.set_state(CustomerAccountStates.menu)
        await _show_account_menu(message, state)
        return
    phone = _normalize_phone(message.text)
    if not phone:
        await _send_account_contact_prompt(
            message,
            state,
            "شماره وارد شده معتبر نیست. لطفاً مجدداً تلاش کنید.",
        )
        return
    await _update_registration(state, phone=phone)
    await state.set_state(CustomerAccountStates.register_first)
    await _send_clean_message(
        state,
        message.bot,
        message.chat.id,
        "نام کوچک خود را وارد کنید:",
        types.ReplyKeyboardRemove(),
        "account_prompt_message",
    )


@router.message(CustomerAccountStates.register_first)
async def account_register_first(message: types.Message, state: FSMContext):
    if _is_cancel_text(message.text):
        await state.set_state(CustomerAccountStates.menu)
        await _show_account_menu(message, state)
        return
    if not message.text or len(message.text.strip()) < 2:
        await _send_clean_message(
            state,
            message.bot,
            message.chat.id,
            "نام وارد شده کوتاه است. لطفاً نام معتبر وارد کنید.",
            None,
            "account_prompt_message",
        )
        return
    await _update_registration(state, first_name=message.text.strip())
    await state.set_state(CustomerAccountStates.register_last)
    await _send_clean_message(
        state,
        message.bot,
        message.chat.id,
        "نام خانوادگی را وارد کنید:",
        None,
        "account_prompt_message",
    )


@router.message(CustomerAccountStates.register_last)
async def account_register_last(message: types.Message, state: FSMContext):
    if _is_cancel_text(message.text):
        await state.set_state(CustomerAccountStates.menu)
        await _show_account_menu(message, state)
        return
    if not message.text or len(message.text.strip()) < 2:
        await _send_clean_message(
            state,
            message.bot,
            message.chat.id,
            "نام خانوادگی معتبر نیست. لطفاً دوباره تلاش کنید.",
            None,
            "account_prompt_message",
        )
        return
    await _update_registration(state, last_name=message.text.strip())
    await state.set_state(CustomerAccountStates.register_address)
    await _send_clean_message(
        state,
        message.bot,
        message.chat.id,
        "آدرس کامل محل تحویل را ارسال کنید:",
        None,
        "account_prompt_message",
    )


@router.message(CustomerAccountStates.register_address)
async def account_register_address(message: types.Message, state: FSMContext):
    if _is_cancel_text(message.text):
        await state.set_state(CustomerAccountStates.menu)
        await _show_account_menu(message, state)
        return
    if not message.text or len(message.text.strip()) < 5:
        await _send_clean_message(
            state,
            message.bot,
            message.chat.id,
            "آدرس خیلی کوتاه است؛ لطفاً دقیق‌تر وارد کنید.",
            None,
            "account_prompt_message",
        )
        return
    await _update_registration(state, address=message.text.strip())
    await state.set_state(CustomerAccountStates.register_postal)
    await _send_clean_message(
        state,
        message.bot,
        message.chat.id,
        "کد پستی را ارسال کنید (یا عبارت «رد کردن» را بفرستید):",
        None,
        "account_prompt_message",
    )


@router.message(CustomerAccountStates.register_postal)
async def account_register_postal(message: types.Message, state: FSMContext):
    if _is_cancel_text(message.text):
        await state.set_state(CustomerAccountStates.menu)
        await _show_account_menu(message, state)
        return
    text = (message.text or "").strip()
    if _is_skip_text(text) or not text:
        await _update_registration(state, postal_code=None)
    else:
        await _update_registration(state, postal_code=text)
    await state.set_state(CustomerAccountStates.register_location)
    await _send_clean_message(
        state,
        message.bot,
        message.chat.id,
        "لوکیشن محل تحویل را ارسال کنید یا عبارت «رد کردن» را بفرستید:",
        None,
        "account_prompt_message",
    )


@router.message(CustomerAccountStates.register_location, F.location)
async def account_register_location(message: types.Message, state: FSMContext):
    await _update_registration(
        state,
        delivery_lat=message.location.latitude,
        delivery_lng=message.location.longitude,
    )
    await state.set_state(CustomerAccountStates.register_password)
    await _send_clean_message(
        state,
        message.bot,
        message.chat.id,
        "رمز عبور دلخواه را ارسال کنید:",
        None,
        "account_prompt_message",
    )


@router.message(CustomerAccountStates.register_location)
async def account_register_location_text(message: types.Message, state: FSMContext):
    if _is_cancel_text(message.text):
        await state.set_state(CustomerAccountStates.menu)
        await _show_account_menu(message, state)
        return
    if _is_skip_text(message.text):
        await _update_registration(state, delivery_lat=None, delivery_lng=None)
        await state.set_state(CustomerAccountStates.register_password)
        await _send_clean_message(
            state,
            message.bot,
            message.chat.id,
            "رمز عبور دلخواه را ارسال کنید:",
            None,
            "account_prompt_message",
        )
        return
    await _send_clean_message(
        state,
        message.bot,
        message.chat.id,
        "برای ثبت لوکیشن، موقعیت جغرافیایی ارسال کنید یا عبارت «رد کردن» را بفرستید.",
        None,
        "account_prompt_message",
    )


@router.message(CustomerAccountStates.register_password)
async def account_register_password(message: types.Message, state: FSMContext):
    await _update_registration(state, password=message.text.strip())
    data = await state.get_data()
    reg: dict[str, Any] = data.get("account_register") or {}
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        await message.answer("شناسه فروشگاه تنظیم نشده است.")
        await state.set_state(CustomerAccountStates.menu)
        await _show_account_menu(message, state)
        return
    payload = {
        "first_name": reg.get("first_name"),
        "last_name": reg.get("last_name"),
        "phone": reg.get("phone"),
        "address": reg.get("address"),
        "postal_code": reg.get("postal_code"),
        "delivery_lat": reg.get("delivery_lat"),
        "delivery_lng": reg.get("delivery_lng"),
        "password": reg.get("password"),
    }
    # location fields are optional; postal_code is also optional
    missing_optional = {"postal_code", "delivery_lat", "delivery_lng"}
    missing = [k for k, v in payload.items() if v in (None, "", 0) and k not in missing_optional]
    if missing:
        await _send_clean_message(
            state,
            message.bot,
            message.chat.id,
            "اطلاعات وارد شده کامل نیست. لطفاً مراحل را از ابتدا انجام دهید.",
            keyboards.customer_account_menu(False),
            "account_prompt_message",
        )
        await state.set_state(CustomerAccountStates.menu)
        await _show_account_menu(message, state)
        return
    client = BackendClient()
    waiting = await _show_loader(message, "در حال ایجاد حساب کاربری...")
    try:
        await client.post("/public/customers/register", payload, tenant_id=tenant_id)
    except Exception as exc:
        await waiting.edit_text(f"ثبت حساب ناموفق بود: {exc}")
        return
    try:
        auth = await client.login(payload["phone"], payload["password"])
    except Exception as exc:
        await waiting.edit_text(f"ورود خودکار ناموفق بود: {exc}")
        return
    role_label = _normalize_role(auth.get("role"))
    await state.update_data(
        cust_token=auth["token"],
        cust_tenant_id=auth.get("tenant_id"),
        account_register=None,
        user_role=role_label,
    )
    _session_update(
        message.from_user.id,
        token=auth["token"],
        tenant_id=auth.get("tenant_id"),
        role=role_label,
        token_scope="customer",
    )
    await state.set_state(CustomerAccountStates.menu)
    await waiting.edit_text("حساب کاربری ایجاد و ورود انجام شد.")
    await _show_account_menu(message, state)


# ---------------------------------------------------------------------------
# Customer flow
# ---------------------------------------------------------------------------


@router.message(F.text == "📢 مشاهده محصولات")
async def customer_view_products(message: types.Message, state: FSMContext):
    await state.set_state(CustomerFlowStates.menu)
    success = await _handle_channel_entry(message, state, message.from_user.id)
    if not success:
        await message.answer("برای مشاهده محصولات، لطفاً ابتدا مشتری پیش‌فرض فروشگاه را تنظیم کنید.")
        return
    data = await state.get_data()
    await message.answer("منوی فروشگاه:", reply_markup=_customer_menu_markup(data))


@router.callback_query(F.data == "cust:menu")
async def customer_menu(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    await state.set_state(CustomerFlowStates.menu)
    await callback.answer()
    await callback.message.edit_text("منوی فروشگاه:", reply_markup=_customer_menu_markup(data))


@router.callback_query(F.data == "cust:back")
async def customer_back(callback: types.CallbackQuery, state: FSMContext):
    await state.clear()
    await callback.answer()
    data = await state.get_data()
    await callback.message.edit_text("به صفحه اصلی بازگشتید.", reply_markup=_customer_menu_markup(data))


@router.callback_query(F.data == "cust:favorites")
async def customer_favorites(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        await callback.answer("تنظیم STORE_TENANT_ID الزامی است.", show_alert=True)
        return
    token = await _get_customer_token(state, callback.from_user.id)
    if not token:
        await callback.answer("ورود ناموفق بود. لطفاً دوباره تلاش کنید.", show_alert=True)
        return
    try:
        favorites = await _load_customer_favorites(tenant_id, token, callback.from_user.id)
    except Exception as exc:
        await callback.answer(f"خطا در دریافت علاقه‌مندی‌ها: {exc}", show_alert=True)
        return
    await callback.answer()
    await _render_customer_favorites(callback, state, favorites)


@router.callback_query(F.data.startswith("fav:remove:"))
async def favorite_remove(callback: types.CallbackQuery, state: FSMContext):
    product_id = callback.data.split(":", 2)[2]
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        await callback.answer("تنظیم STORE_TENANT_ID الزامی است.", show_alert=True)
        return
    token = await _get_customer_token(state, callback.from_user.id)
    if not token:
        await callback.answer("برای ادامه باید وارد شوید.", show_alert=True)
        return
    client = BackendClient(token=token, telegram_id=callback.from_user.id)
    try:
        await client.delete(f"/visits/me/likes/{product_id}", tenant_id=tenant_id)
    except Exception as exc:
        await callback.answer(f"حذف علاقه‌مندی موفق نبود: {exc}", show_alert=True)
        return
    await callback.answer("محصول از علاقه‌مندی‌ها حذف شد.", show_alert=True)
    try:
        favorites = await _load_customer_favorites(tenant_id, token, callback.from_user.id)
    except Exception as exc:
        await callback.answer(f"خطا در به‌روزرسانی لیست: {exc}", show_alert=True)
        return
    await _render_customer_favorites(callback, state, favorites)


@router.callback_query(F.data.startswith("fav:cart:"))
async def favorite_add_to_cart(callback: types.CallbackQuery, state: FSMContext):
    product_id = callback.data.split(":", 2)[2]
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        await callback.answer("تنظیم STORE_TENANT_ID الزامی است.", show_alert=True)
        return
    token = await _get_customer_token(state, callback.from_user.id)
    if not token:
        await callback.answer("برای ادامه باید وارد شوید.", show_alert=True)
        return
    client = BackendClient(token=token, telegram_id=callback.from_user.id)
    try:
        await client.post(
            "/cart/items",
            {"product_id": product_id, "qty": 1},
            tenant_id=tenant_id,
        )
    except httpx.HTTPStatusError as exc:
        await callback.answer(f"افزودن به سبد موفق نبود: {exc}", show_alert=True)
        return
    except Exception as exc:
        await callback.answer(f"افزودن به سبد موفق نبود: {exc}", show_alert=True)
        return
    await callback.answer("محصول به سبد خرید اضافه شد.", show_alert=True)
    try:
        favorites = await _load_customer_favorites(tenant_id, token, callback.from_user.id)
    except Exception as exc:
        await callback.answer(f"خطا در به‌روزرسانی لیست: {exc}", show_alert=True)
        return
    await _render_customer_favorites(callback, state, favorites)


@router.callback_query(F.data == "cust:login")
async def customer_login(callback: types.CallbackQuery, state: FSMContext):
    data_before = await state.get_data()
    tenant_id = _customer_tenant_id(data_before)
    if not tenant_id:
        await callback.answer("STORE_TENANT_ID تنظیم نشده است.", show_alert=True)
        return
    client = BackendClient(telegram_id=callback.from_user.id)
    try:
        auth = await client.post("/auth/telegram-login", {}, tenant_id=tenant_id)
    except Exception as exc:
        await callback.answer(f"ورود ناموفق بود: {exc}", show_alert=True)
        return
    role_label = _normalize_role(auth.get("role"))
    updates = {
        "cust_token": auth["token"],
        "cust_tenant_id": auth["tenant_id"],
        "user_role": role_label,
    }
    if _is_admin_role(role_label):
        updates["token"] = auth["token"]
        updates["tenant_id"] = auth["tenant_id"]
    await state.update_data(**updates)
    _session_update(
        callback.from_user.id,
        token=auth["token"],
        tenant_id=auth["tenant_id"],
        role=role_label,
        token_scope="customer",
    )
    await callback.answer("ورود موفق بود.", show_alert=True)
    if _is_admin_role(role_label):
        await state.set_state(StoreLoginStates.admin_menu)
        await callback.message.answer("شما در نقش مدیر وارد شدید.", reply_markup=keyboards.admin_main_menu())
        return
    data = await state.get_data()
    await callback.message.edit_reply_markup(reply_markup=_customer_menu_markup(data))


@router.callback_query(F.data == "cust:enter_channel")
async def customer_enter_channel(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    if _is_admin_role(data.get("user_role")):
        await callback.answer()
        await callback.message.answer("شما در نقش مدیر هستید. از منوی مدیریت استفاده کنید.", reply_markup=keyboards.admin_main_menu())
        return
    success = await _handle_channel_entry(callback, state, callback.from_user.id)
    if success:
        data = await state.get_data()
        await callback.message.answer("منوی فروشگاه:", reply_markup=_customer_menu_markup(data))
    else:
        await callback.answer("برای ورود به کانال، ابتدا از منوی ربات گزینه «مشاهده محصولات» را بزنید.", show_alert=True)


@router.callback_query(F.data == "cust:end_visit")
async def customer_end_visit(callback: types.CallbackQuery, state: FSMContext):
    likes, ok = await _complete_customer_visit(state, callback.from_user.id)
    if not ok:
        await callback.answer("بازدید فعالی در جریان نیست.", show_alert=True)
        return
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    selection = await _build_selection_from_likes(likes, tenant_id)
    await state.update_data(cust_selection=selection)
    summary = _format_likes_summary(likes)
    await callback.answer()
    await callback.message.answer(summary)
    await callback.message.answer(_selection_text(selection), reply_markup=_selection_keyboard(selection))
    added = await _commit_selection_to_cart(state, callback.from_user.id, selection, tenant_id)
    if added:
        await callback.message.answer("محصولات لایک‌شده به سبد خرید اضافه شدند.")
    data = await state.get_data()
    await callback.message.answer("منوی فروشگاه:", reply_markup=_customer_menu_markup(data))


@router.callback_query(F.data == "cust:browse")
async def customer_browse(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        await callback.answer("STORE_TENANT_ID تنظیم نشده است.", show_alert=True)
        return
    client = BackendClient()
    cats = await client.get("/categories", tenant_id=tenant_id)
    logged = bool(data.get("cust_token"))
    text = "\n".join(f"- {c['name']}" for c in cats) or "دسته‌بندی یافت نشد."
    await state.set_state(CustomerFlowStates.browsing)
    await callback.answer()
    await callback.message.edit_text(text, reply_markup=keyboards.customer_categories_kb(cats, logged))


@router.callback_query(F.data.startswith("cust:cat:"))
async def customer_category(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        await callback.answer("STORE_TENANT_ID تنظیم نشده است.", show_alert=True)
        return
    cat_id = callback.data.split(":", 2)[2]
    client = BackendClient()
    prods = await client.get("/products", tenant_id=tenant_id, params={"category_id": cat_id, "active": True})
    logged = bool(data.get("cust_token"))
    text = "\n".join(f"- {p['title']}" for p in prods) or "محصولی موجود نیست."
    await state.set_state(CustomerFlowStates.products)
    await callback.answer()
    await callback.message.edit_text(text, reply_markup=keyboards.customer_products_kb(prods, logged))


@router.callback_query(F.data.startswith("cust:add:"))
async def customer_add_product(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    token = data.get("cust_token")
    tenant_id = _customer_tenant_id(data)
    if not (token and tenant_id):
        await callback.answer("برای خرید ابتدا وارد شوید.", show_alert=True)
        return
    product_id = callback.data.split(":", 2)[2]
    client = BackendClient(token=token, telegram_id=callback.from_user.id)
    await client.post("/cart/items", {"product_id": product_id, "qty": 1}, tenant_id=tenant_id)
    await callback.answer("به سبد خرید اضافه شد.")


@router.callback_query(F.data == "cust:cart")
async def customer_cart(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        await callback.answer("STORE_TENANT_ID تنظیم نشده است.", show_alert=True)
        return
    token = await _get_customer_token(state, callback.from_user.id)
    if not token:
        await callback.answer("برای مشاهده سبد ابتدا وارد شوید.", show_alert=True)
        return
    client = BackendClient(token=token, telegram_id=callback.from_user.id)
    order = await client.post("/cart/start", {}, tenant_id=tenant_id)
    await state.set_state(CustomerFlowStates.cart)
    await callback.answer()
    await _render_cart_summary(
        state,
        callback.message.bot,
        callback.message.chat.id,
        order,
        tenant_id,
        token,
        callback.from_user.id,
    )


@router.callback_query(F.data.startswith("cart:item:remove:"))
async def customer_cart_item_remove(callback: types.CallbackQuery, state: FSMContext):
    item_id = callback.data.split(":", 3)[3]
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        await callback.answer("STORE_TENANT_ID تنظیم نشده است.", show_alert=True)
        return
    token = await _get_customer_token(state, callback.from_user.id)
    if not token:
        await callback.answer("برای ادامه باید وارد شوید.", show_alert=True)
        return
    client = BackendClient(token=token, telegram_id=callback.from_user.id)
    try:
        order = await client.delete(f"/cart/items/{item_id}", tenant_id=tenant_id)
    except Exception as exc:
        await callback.answer(f"حذف امکان‌پذیر نبود: {exc}", show_alert=True)
        return
    await callback.answer("محصول از سبد حذف شد.", show_alert=True)
    await state.update_data(cart_pending_item_id=None)
    await _render_cart_summary(
        state,
        callback.message.bot,
        callback.message.chat.id,
        order,
        tenant_id,
        token,
        callback.from_user.id,
    )


@router.callback_query(F.data.startswith("cart:item:complete:"))
async def customer_cart_item_complete(callback: types.CallbackQuery, state: FSMContext):
    item_id = callback.data.split(":", 3)[3]
    await state.update_data(cart_pending_item_id=item_id)
    await state.set_state(CustomerFlowStates.awaiting_cart_quantity)
    await callback.answer()
    await callback.message.answer(
        "تعداد مورد نظر برای آن محصول را وارد کنید. برای انصراف /cancel را ارسال کنید."
    )


@router.callback_query(F.data == "cust:cart_cancel")
async def customer_cart_cancel(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    token = data.get("cust_token")
    tenant_id = _customer_tenant_id(data)
    if not (token and tenant_id):
        await callback.answer("ابتدا وارد شوید.", show_alert=True)
        return
    client = BackendClient(token=token, telegram_id=callback.from_user.id)
    await client.post("/cart/cancel", {}, tenant_id=tenant_id)
    await state.update_data(last_cart_message_id=None, cart_pending_item_id=None)
    await callback.answer("سبد حذف شد.", show_alert=True)


@router.message(CustomerFlowStates.awaiting_cart_quantity)
async def customer_cart_quantity(message: types.Message, state: FSMContext):
    text = (message.text or "").strip()
    if _is_cancel_text(text):
        await state.update_data(cart_pending_item_id=None)
        await state.set_state(CustomerFlowStates.cart)
        await message.answer("عملیات لغو شد.")
        return
    try:
        qty = int(text)
    except ValueError:
        await message.answer("لطفاً یک عدد صحیح وارد کنید.")
        return
    if qty <= 0:
        await message.answer("تعداد باید بزرگ‌تر از صفر باشد.")
        return
    data = await state.get_data()
    item_id = data.get("cart_pending_item_id")
    tenant_id = _customer_tenant_id(data)
    if not (item_id and tenant_id):
        await message.answer("امکان بروزرسانی سبد وجود ندارد.")
        return
    token = await _get_customer_token(state, message.from_user.id)
    if not token:
        await message.answer("برای ادامه باید وارد شوید.")
        return
    client = BackendClient(token=token, telegram_id=message.from_user.id)
    try:
        order = await client.put(f"/cart/items/{item_id}", {"qty": qty}, tenant_id=tenant_id)
    except Exception as exc:
        await message.answer(f"به‌روزرسانی امکان‌پذیر نیست: {exc}")
        return
    await state.set_state(CustomerFlowStates.cart)
    await state.update_data(cart_pending_item_id=None)
    await _render_cart_summary(
        state,
        message.bot,
        message.chat.id,
        order,
        tenant_id,
        token,
        message.from_user.id,
    )


@router.callback_query(F.data == "cust:checkout")
async def customer_checkout(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        await callback.answer("STORE_TENANT_ID تنظیم نشده است.", show_alert=True)
        return
    token = await _get_customer_token(state, callback.from_user.id)
    if not token:
        await callback.answer("ابتدا وارد شوید.", show_alert=True)
        return
    await state.set_state(CustomerFlowStates.checkout)
    await callback.answer()
    await callback.message.answer(
        "برای ادامه روش پرداخت را انتخاب کنید:",
        reply_markup=keyboards.payment_methods_keyboard(),
    )


@router.callback_query(F.data == "payment:card2card")
async def payment_card2card(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    tenant_id = _customer_tenant_id(data)
    if not tenant_id:
        await callback.answer("STORE_TENANT_ID تنظیم نشده است.", show_alert=True)
        return
    token = await _get_customer_token(state, callback.from_user.id)
    if not token:
        await callback.answer("برای ادامه باید وارد شوید.", show_alert=True)
        return
    client = BackendClient(token=token, telegram_id=callback.from_user.id)
    try:
        order = await client.post("/cart/checkout", {"payment_method": "card2card"}, tenant_id=tenant_id)
    except Exception as exc:
        await callback.answer(f"خطا در ثبت سفارش: {exc}", show_alert=True)
        return
    profile = await _load_store_profile(tenant_id)
    manager_ids = _normalize_manager_ids(profile.get("manager_telegram_ids") if profile else None)
    if not manager_ids:
        manager_ids = _parse_admin_ids(settings.HYPER_ADMIN_TELEGRAM_IDS)
    if not manager_ids:
        logger.warning("no manager Telegram IDs configured for tenant %s", tenant_id)
    await state.update_data(
        pending_receipt_order_id=order.get("id"),
        pending_receipt_amount=order.get("total"),
        pending_receipt_manager_ids=manager_ids,
        pending_receipt_tenant_id=tenant_id,
    )
    await state.set_state(CustomerFlowStates.waiting_receipt)
    amount = order.get("total") or 0
    card_note = (profile or {}).get("payment_link_note")
    message_lines = [
        f"مبلغ قابل پرداخت: {amount:,.0f}",
        f"شماره کارت فروشگاه: {card_note or 'ثبت نشده'}",
        "پس از واریز، رسید کارت‌به‌کارت را به صورت عکس ارسال کنید تا سفارش تایید شود.",
    ]
    await callback.answer()
    await callback.message.answer("\n".join(message_lines), reply_markup=keyboards.receipt_waiting_keyboard())


@router.callback_query(F.data == "payment:online")
async def payment_online(callback: types.CallbackQuery, state: FSMContext):
    await callback.answer("این روش در حال حاضر غیرفعال است.", show_alert=True)


@router.message(CustomerFlowStates.waiting_receipt, F.photo)
async def customer_receipt_photo(message: types.Message, state: FSMContext):
    data = await state.get_data()
    order_id = data.get("pending_receipt_order_id")
    amount = data.get("pending_receipt_amount") or 0
    manager_ids = data.get("pending_receipt_manager_ids") or []
    if not manager_ids:
        manager_ids = _parse_admin_ids(settings.HYPER_ADMIN_TELEGRAM_IDS)
    if not order_id:
        await message.answer("سفارش معتبر یافت نشد.")
        return
    report_text = (
        "سفارش جدید توسط مشتری ثبت شد.\n"
        f"سفارش: {order_id[:8]}\n"
        f"مبلغ: {amount:,.0f}\n"
        f"مشتری: {message.from_user.full_name} (@{message.from_user.username or 'بدون نام'})\n"
        f"‎Telegram ID: {message.from_user.id}"
    )
    for manager_id in manager_ids:
        try:
            await message.bot.send_message(manager_id, report_text)
            await message.bot.forward_message(manager_id, message.chat.id, message.message_id)
        except Exception as exc:
            logger.warning("failed to report receipt to %s: %s", manager_id, exc)
    await state.update_data(
        pending_receipt_order_id=None,
        pending_receipt_amount=None,
        pending_receipt_manager_ids=None,
        pending_receipt_tenant_id=None,
    )
    await state.set_state(CustomerFlowStates.menu)
    await _send_clean_message(
        state,
        message.bot,
        message.chat.id,
        "سفارش شما ثبت شد. از منوی فروشگاه استفاده کنید.",
        _customer_menu_markup(await state.get_data()),
        "last_bot_message_id",
    )


@router.message(CustomerFlowStates.waiting_receipt)
async def customer_receipt_prompt(message: types.Message, state: FSMContext):
    await message.answer("لطفاً رسید پرداخت را به صورت عکس ارسال کنید.")


@router.message(CustomerFlowStates.waiting_receipt, Command("cancel"))
async def customer_receipt_cancel(message: types.Message, state: FSMContext):
    await state.update_data(
        pending_receipt_order_id=None,
        pending_receipt_amount=None,
        pending_receipt_manager_ids=None,
        pending_receipt_tenant_id=None,
    )
    await state.set_state(CustomerFlowStates.cart)
    await message.answer("عملیات پرداخت لغو شد. می‌توانید دوباره به سبد برگردید.")


@router.callback_query(F.data == "cust:edit_cart")
async def customer_edit_cart(callback: types.CallbackQuery, state: FSMContext):
    selection: list[dict] = (await state.get_data()).get("cust_selection") or []
    if not selection:
        await callback.answer("فهرست انتخابی خالی است.", show_alert=True)
        return
    await callback.answer()
    await callback.message.edit_text(_selection_text(selection), reply_markup=_selection_keyboard(selection))


@router.callback_query(F.data.startswith("cust:sel:"))
async def customer_selection_ops(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    selection: list[dict] = data.get("cust_selection") or []
    parts = callback.data.split(":")
    action = parts[2]
    if action == "noop":
        await callback.answer()
        return
    if action == "clear":
        selection = []
        await state.update_data(cust_selection=selection)
        await callback.answer("فهرست حذف شد.", show_alert=True)
        await callback.message.edit_text(_selection_text(selection), reply_markup=_selection_keyboard(selection))
        return
    if action == "commit":
        token = data.get("cust_token")
        tenant_id = _customer_tenant_id(data)
        if not (token and tenant_id):
            await callback.answer("برای افزودن به سبد ابتدا وارد شوید.", show_alert=True)
            return
        client = BackendClient(token=token, telegram_id=callback.from_user.id)
        for item in selection:
            await client.post(
                "/cart/items",
                {"product_id": item["product_id"], "qty": item["qty"]},
                tenant_id=tenant_id,
            )
        await callback.answer("به سبد خرید اضافه شد.", show_alert=True)
        return
    pid = parts[3] if len(parts) > 3 else None
    updated = False
    for item in selection:
        if item["product_id"] == pid:
            if action == "inc":
                item["qty"] += 1
            elif action == "dec":
                item["qty"] = max(1, item["qty"] - 1)
            elif action == "remove":
                selection.remove(item)
            updated = True
            break
    await state.update_data(cust_selection=selection)
    await callback.answer("به‌روزرسانی شد." if updated else "آیتم یافت نشد.", show_alert=not updated)
    await callback.message.edit_text(_selection_text(selection), reply_markup=_selection_keyboard(selection))


@router.callback_query(F.data.startswith("chan:like"))
async def channel_like_callback(callback: types.CallbackQuery, state: FSMContext):
    """ثبت پسند (لایک) محصول از پیام کانال."""
    # داده‌ی callback به صورت chan:like:<product_id> است
    product_id = callback.data.split(":", 2)[2]

    # ۱) چک فرمت UUID محصول؛ اگر خراب باشد اصلاً درخواست به سرور نرود
    try:
        product_uuid = str(uuid.UUID(product_id))
    except ValueError:
        await callback.answer(
            "این پیام به یک محصول معتبر متصل نیست. لطفاً از پیام‌های جدید ربات استفاده کن.",
            show_alert=True,
        )
        return

    session = _session_get(callback.from_user.id)
    tenant_id = session.get("tenant_id") if session else None
    session_token = session.get("session_token") if session else None
    if not (tenant_id and session_token):
        await callback.answer(
            "ابتدا از منوی ربات گزینه «📢 مشاهده محصولات» را بزن تا بازدید جدید شروع شود.",
            show_alert=True,
        )
        return

    client = BackendClient()
    try:
        resp = await client.post(
            "/visits/likes",
            {
                "product_id": product_uuid,
                "telegram_user_id": callback.from_user.id,
                "session_token": session_token,
            },
            tenant_id=tenant_id,
        )
    except httpx.HTTPStatusError as exc:
        # اگر بازدید منقضی شده یا پیدا نشد
        if exc.response is not None and exc.response.status_code == 404:
            await callback.answer(
                "بازدید فعلی پیدا نشد یا منقضی شده است. "
                "لطفاً از منوی ربات دوباره «📢 مشاهده محصولات» را بزن.",
                show_alert=True,
            )
            _session_update(callback.from_user.id, session_token=None)
            return
        await callback.answer(f"خطا در ثبت پسند: {exc}", show_alert=True)
        return
    except Exception as exc:
        await callback.answer(f"خطا در ثبت پسند: {exc}", show_alert=True)
        return

    await callback.answer("ثبت شد ✅", show_alert=False)


@router.callback_query(F.data.startswith("chan:add"))
async def channel_add_to_cart(callback: types.CallbackQuery, state: FSMContext):
    """افزودن محصول از پیام کانال به سبد خرید مشتری."""
    product_id = callback.data.split(":", 2)[2]

    # ۱) چک فرمت UUID؛ اگر معتبر نباشد، 422 از سرور می‌گیریم
    try:
        product_uuid = str(uuid.UUID(product_id))
    except ValueError:
        await callback.answer(
            "این دکمه به یک محصول معتبر متصل نیست. "
            "لطفاً از پیامی که ربات همین الان به کانال ارسال کرده استفاده کن.",
            show_alert=True,
        )
        return

    session = _session_get(callback.from_user.id)
    tenant_id = session.get("tenant_id") if session else None
    token = _session_customer_token(session)
    if not (tenant_id and token):
        await callback.answer(
            "برای افزودن به سبد ابتدا از منوی ربات وارد حساب کاربری شو.",
            show_alert=True,
        )
        return

    client = BackendClient(token=token, telegram_id=callback.from_user.id)
    try:
        await client.post(
            "/cart/items",
            {"product_id": product_uuid, "qty": 1, "chosen_specs": {}},
            tenant_id=tenant_id,
        )
    except httpx.HTTPStatusError as exc:
        if exc.response is not None and exc.response.status_code == 422:
            # تلاش برای خواندن پیام توضیحی از سرور
            detail = None
            try:
                detail_json = exc.response.json()
                if isinstance(detail_json, dict):
                    detail = detail_json.get("detail")
            except Exception:
                detail = None

            msg = "افزودن به سبد ناموفق بود."
            if isinstance(detail, str):
                msg += f"\n{detail}"
            await callback.answer(msg, show_alert=True)
            return

        await callback.answer(f"خطا در افزودن به سبد: {exc}", show_alert=True)
        return
    except Exception as exc:
        await callback.answer(f"خطا در افزودن به سبد: {exc}", show_alert=True)
        return

    await callback.answer("به سبد اضافه شد ✅", show_alert=False)


@router.callback_query(F.data.startswith("chan:checkout"))
async def channel_checkout_callback(callback: types.CallbackQuery, state: FSMContext):
    session = _session_get(callback.from_user.id)
    tenant_id = session.get("tenant_id") if session else None
    session_token = session.get("session_token") if session else None
    if not (tenant_id and session_token):
        await callback.answer("ابتدا از منوی ربات بازدید را آغاز کنید.", show_alert=True)
        return
    await callback.answer("در حال آماده‌سازی فهرست خرید...", show_alert=True)
    await _finalize_checkout_summary(
        state,
        callback.from_user.id,
        tenant_id,
        session_token,
        callback.bot,
        callback.from_user.id,
    )
