from __future__ import annotations
import os
import sys
from logging.config import fileConfig
from sqlalchemy import engine_from_config, pool
from alembic import context

# Ensure project root is on sys.path
BASE_DIR = os.path.abspath(os.path.join(os.path.dirname(__file__), os.pardir, os.pardir))
if BASE_DIR not in sys.path:
    sys.path.insert(0, BASE_DIR)

# Optional: load .env so DB_DSN is available
try:  # pragma: no cover
    from dotenv import load_dotenv  # type: ignore
    load_dotenv(os.path.join(BASE_DIR, ".env"))
except Exception:
    pass

from app.db.base import Base  # noqa: E402
import app.models  # noqa: F401,E402  (populate metadata)


config = context.config
# Prefer DB_DSN from environment (or .env)
if os.getenv("DB_DSN"):
    config.set_main_option("sqlalchemy.url", os.environ["DB_DSN"])

if config.config_file_name is not None:
    fileConfig(config.config_file_name)

    target_metadata = Base.metadata

    # Ensure URL is present after overrides
    url_from_cfg = config.get_main_option("sqlalchemy.url")
    if not url_from_cfg:
        raise RuntimeError("DB_DSN/sqlalchemy.url is not set. Set DB_DSN in environment or alembic.ini.")


def run_migrations_offline():
    url = config.get_main_option("sqlalchemy.url")
    context.configure(
        url=url,
        target_metadata=target_metadata,
        literal_binds=True,
        dialect_opts={"paramstyle": "named"},
    )

    with context.begin_transaction():
        context.run_migrations()


def run_migrations_online():
    connectable = engine_from_config(
        config.get_section(config.config_ini_section),
        prefix="sqlalchemy.",
        poolclass=pool.NullPool,
    )

    with connectable.connect() as connection:
        context.configure(connection=connection, target_metadata=target_metadata)

        with context.begin_transaction():
            context.run_migrations()


def run_migrations():
    if context.is_offline_mode():
        run_migrations_offline()
    else:
        run_migrations_online()


run_migrations()

